package com.bxm.localnews.integration;

import com.alibaba.fastjson.JSON;
import com.bxm.egg.facade.dto.LocationDetailDTO;
import com.bxm.egg.facade.service.LocationFacadeService;
import com.bxm.localnews.news.dto.LocationDTO;
import com.bxm.localnews.news.model.dto.AreaCode;
import com.bxm.localnews.news.vo.Location;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.stereotype.Service;

import javax.annotation.PostConstruct;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.TimeUnit;

@Slf4j
@Service
public class LocationIntegrationService {

    private LoadingCache<String, LocationDTO> cache;

    private final LocationFacadeService locationFacadeService;

    public LocationIntegrationService(InnerServiceIntegrationService innerServiceIntegrationService) {
        locationFacadeService = innerServiceIntegrationService.getLocationFacadeService();
    }

    @PostConstruct
    public void buildCache() {
        cache = CacheBuilder.newBuilder()
                .maximumSize(2000L)
                .expireAfterWrite(10L, TimeUnit.MINUTES)
                .build(new CacheLoader<String, LocationDTO>() {
                    @Override
                    public LocationDTO load(String key) {
                        LocationDetailDTO detailDTO = locationFacadeService.getLocationDetailByCode(key);

                        LocationDTO location = new LocationDTO();
                        if (null == detailDTO) {
                            location.setCode(key);
                            location.setName("未知");
                        } else {
                            location.setCode(key);
                            location.setName(detailDTO.getName());
                        }

                        return location;
                    }
                });
    }

    public LocationDTO getLocationByGeocode(String code) {
        if (StringUtils.isBlank(code)) {
            return new LocationDTO();
        }

        return cache.getUnchecked(code);
    }

    /**
     * 批量获取地区信息，兼容管理后台之前的逻辑
     *
     * @param codes 地区编码列表
     * @return 地区信息列表
     */
    public List<Location> batchFetch(List<String> codes) {
        List<Location> locations = Lists.newArrayList();

        for (String locationCode : codes) {
            LocationDTO locationDTO = getLocationByGeocode(locationCode);
            locations.add(convert(locationDTO));
        }

        return locations;
    }

    private Location convert(LocationDTO locationDTO) {
        Location location = new Location();
        BeanUtils.copyProperties(locationDTO, location);
        return location;
    }

    public String batchGetDetailJson(String[] areaCodes) {
        List<AreaCode> areaCodeList = new ArrayList<>();

        for (String areaCode : areaCodes) {
            LocationDTO locationDTO = getLocationByGeocode(areaCode);

            AreaCode areaInfo = new AreaCode();
            areaInfo.setAreaCode(locationDTO.getCode());
            areaInfo.setAreaName(locationDTO.getName());

            areaCodeList.add(areaInfo);
        }


        return JSON.toJSONString(areaCodeList);
    }
}























