package com.bxm.localnews.integration;

import com.bxm.egg.common.constant.AppConst;
import com.bxm.foundation.user.facade.VirtualUserFacadeService;
import com.bxm.foundation.user.facade.dto.VirtualUserDTO;
import com.bxm.foundation.user.facade.param.VirtualUserQueryParam;
import com.bxm.localnews.news.model.dto.VirtualUserOverviewDTO;
import com.bxm.localnews.news.model.param.VirtualUserRandomQueryParam;
import com.bxm.localnews.news.model.vo.VirtualUser;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import java.util.List;
import java.util.Objects;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * @author liujia
 * @date 2020-08-06 18:28
 **/
@Slf4j
@Service
public class VirtualUserIntegrationService {

    private VirtualUserFacadeService virtualUserFacadeService;

    public VirtualUserIntegrationService(InnerServiceIntegrationService innerServiceIntegrationService) {
        virtualUserFacadeService = innerServiceIntegrationService.getVirtualUserFacadeService();
    }

    private LoadingCache<Long, VirtualUserOverviewDTO> cache;

    /**
     * 判断用户是否虚拟用户
     * 简化逻辑，负数的ID就是虚拟用户
     *
     * @param userId 用户ID
     * @return true表示虚拟用户
     */
    public Boolean exists(Long userId) {
        return userId < 0;
    }

    public List<VirtualUser> getRandom(VirtualUserRandomQueryParam param) {
        VirtualUserQueryParam facadeParam = new VirtualUserQueryParam();
        facadeParam.setAcquireNum(param.getNum());
        facadeParam.setSrcApp(AppConst.SRC_APP);

        List<VirtualUserDTO> randomList = virtualUserFacadeService.getRandom(facadeParam);

        return randomList.stream().map(this::of).collect(Collectors.toList());
    }

    private VirtualUser of(VirtualUserDTO userDTO) {
        VirtualUser virtualUser = new VirtualUser();
        virtualUser.setId(userDTO.getUserId());
        virtualUser.setHeadImg(userDTO.getHeadImg());
        virtualUser.setNickname(userDTO.getNickName());
        virtualUser.setSex(userDTO.getSex());
        return virtualUser;
    }

    private VirtualUserOverviewDTO convert(VirtualUserDTO virtualUserDTO) {
        VirtualUserOverviewDTO virtualUserOverviewDTO = new VirtualUserOverviewDTO();
        virtualUserOverviewDTO.setId(virtualUserDTO.getUserId());
        virtualUserOverviewDTO.setNickname(virtualUserDTO.getNickName());
        virtualUserOverviewDTO.setHeadImg(virtualUserDTO.getHeadImg());
        virtualUserOverviewDTO.setUserInfo(virtualUserDTO.getTag());

        return virtualUserOverviewDTO;
    }

    /**
     * 获取单个马甲号的信息，优先查询缓存
     * 为空的话返回空对象
     *
     * @param virtualId 马甲号id
     * @return 马甲号信息
     */
    public VirtualUserOverviewDTO getByIdCacheFirst(Long virtualId) {
        if (Objects.isNull(cache)) {
            cache = CacheBuilder.newBuilder()
                    .maximumSize(1000)
                    .expireAfterAccess(1, TimeUnit.HOURS)
                    .build(new CacheLoader<Long, VirtualUserOverviewDTO>() {
                        @Override
                        public VirtualUserOverviewDTO load(Long key) throws Exception {
                            VirtualUserDTO virtualUserDTO = virtualUserFacadeService.get(key);
                            if (null == virtualUserDTO) {
                                return new VirtualUserOverviewDTO();
                            }
                            return convert(virtualUserDTO);
                        }
                    });
        }
        return cache.getUnchecked(virtualId);
    }
}
