package com.bxm.egg.user.service.barrels.weight;

import com.bxm.egg.user.constant.RedisConfig;
import com.bxm.egg.user.enums.RecommendTypeEnum;
import com.bxm.egg.user.mapper.RecommendNativeMapper;
import com.bxm.egg.user.param.NativeRecommendContext;
import com.bxm.egg.user.param.RecommendNativeParam;
import com.bxm.egg.user.param.WeightParam;
import com.bxm.egg.user.properties.RecommendProperties;
import com.bxm.egg.user.service.barrels.AbstractBarrelChoose;
import com.bxm.egg.user.vo.RecommendNative;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisListAdapter;
import com.bxm.newidea.component.tools.DateUtils;
import com.google.common.collect.Multimap;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.stream.Collectors;

import static org.apache.commons.lang3.RandomUtils.nextInt;

/**
 * 有关分值策略的一种方式
 */
public abstract class AbstractWeightBarrelChoose extends AbstractBarrelChoose {

    @Autowired
    private RecommendNativeMapper recommendNativeMapper;

    @Autowired
    private RedisListAdapter redisListAdapter;

    @Autowired
    private RecommendProperties recommendProperties;

    protected static final String USER_INFO = "user";

    protected static final String RECOMMEND_NATIVE_PARAM = "recommendNativeParam";

    protected static final String USER_BLOCK = "userBlock";

    protected static final String USER_IGNORE = "userIgnore";

    protected static final String USER_CACHE = "userCache";

    /**
     * 得到不同情况的key
     *
     * @param nativeRecommendContext
     * @param strategy
     * @return
     */
    private KeyGenerator getKeyGenerator(NativeRecommendContext nativeRecommendContext, String strategy) {
        return RedisConfig.USER_RECOMMEND.copy()
                .setKey(nativeRecommendContext.getRecommendCategory().toLowerCase())
                .appendKey(strategy)
                .appendKey(nativeRecommendContext.getUserId());
    }

    /**
     * 数据召回
     *
     * @param recommendNativeList
     * @param nativeRecommendContext
     */
    private void processDataRecall(List<RecommendNative> recommendNativeList, NativeRecommendContext nativeRecommendContext) {
        //获取推荐数据之后进行数据找回，降权、拉黑,筛选出拉黑或者降权的用户，不在意这些用户的顺序，所以用并行流
        Map<Long, Integer> blockUserMap = nativeRecommendContext.getParam(USER_BLOCK);
        Multimap<Long, RecommendNative> recommendNativeMap = nativeRecommendContext.getParam(USER_CACHE);
        Map<Long, Integer> ignoreUserMap = nativeRecommendContext.getParam(USER_IGNORE);
        List<RecommendNative> limitUserList = new ArrayList<>();

        Iterator<RecommendNative> it = recommendNativeList.iterator();
        while (it.hasNext()) {
            RecommendNative recommendNative = it.next();
            //忽略的人的处理
            if (!CollectionUtils.isEmpty(ignoreUserMap) && ignoreUserMap.containsKey(recommendNative.getUserId())) {
                it.remove();
                //降权拉黑处理
            } else if (blockUserMap.containsKey(recommendNative.getUserId())) {
                it.remove();
                Integer blockType = blockUserMap.get(recommendNative.getUserId());
                if (2 == blockType) {
                    limitUserList.add(recommendNative);
                }
                //小纸条曝光过的处理
            } else if (recommendNativeMap.containsKey(recommendNative.getUserId())) {
                //当前用户的最后小纸条发布时间
                Long lastNoteId = recommendNative.getLastNoteId();
                //已推荐用户的记录
                Collection<RecommendNative> recommendList = recommendNativeMap.get(recommendNative.getUserId());

                //遍历用户推荐的记录
                for (RecommendNative r : recommendList) {
                    Long recommendLastNoteId = r.getLastNoteId();
                    //如果都没发小纸条
                    if (null == lastNoteId && null == recommendLastNoteId) {
                        it.remove();
                        break;
                        //或者小纸条id相等
                    } else if (null != lastNoteId
                            && lastNoteId.equals(recommendLastNoteId)) {
                        it.remove();
                    }
                }
            }
        }

        recommendNativeList.addAll(limitUserList);
    }

    /**
     * 搜索同城的人之前，先判断是否有，如果没有则填充
     *
     * @param nativeRecommendContext
     * @return
     */
    boolean preSameCityInvoke(NativeRecommendContext nativeRecommendContext) {

        KeyGenerator keyGenerator = getKeyGenerator(nativeRecommendContext, RecommendTypeEnum.SAME_CITY.getName());

        //如果是下拉，需要重新填充
        if (1 == nativeRecommendContext.getActionType()) {

            RecommendNativeParam recommendNativeParam = nativeRecommendContext.getParam(RECOMMEND_NATIVE_PARAM);
            if (null == recommendNativeParam.getLocationCode()) {
                //用户的居住城市都没有，推个屁呀
                nativeRecommendContext.setHasSameCity((byte) 0);
                return false;
            }
            List<RecommendNative> r = recommendNativeMapper.selectByCity(recommendNativeParam);

            //保存到redis中
            redisListAdapter.remove(keyGenerator);

            if (CollectionUtils.isEmpty(r)) {
                //对不起，真的一滴都没有了
                nativeRecommendContext.setHasSameCity((byte) 0);
                return false;
            }

            //获取推荐数据之后进行数据找回，降权、拉黑,筛选出拉黑或者降权的用户，不在意这些用户的顺序，所以用并行流
            this.processDataRecall(r, nativeRecommendContext);

            RecommendNative[] recommendNatives = r.toArray(new RecommendNative[0]);
            redisListAdapter.rightPush(keyGenerator, recommendNatives);
            // 当天结束过期
            redisListAdapter.expire(keyGenerator, DateUtils.getCurSeconds());
        } else {
            int size = redisListAdapter.size(keyGenerator).intValue();
            if (size == 0) {
                //对不起，真的一滴都没有了
                nativeRecommendContext.setHasSameCity((byte) 0);
                return false;
            }
        }

        return true;
    }

    /**
     * 从redis中取出同城的人的列表，如果数量为null，则按比例取
     *
     * @param nativeRecommendContext
     * @param num
     */
    boolean doSameCityInvoke(NativeRecommendContext nativeRecommendContext, Long num) {
        //num如果是null的话则按配置的默认比例来
        if (null == num) {
            WeightParam weightParam = nativeRecommendContext.getParam("weight");
            num = (long) Math.floor(weightParam.getSameCityPercent() * nativeRecommendContext.getPageSize());
        }

        KeyGenerator keyGenerator = getKeyGenerator(nativeRecommendContext, RecommendTypeEnum.SAME_CITY.getName());
        //得到缓存本地人的数量
        Long size = redisListAdapter.size(keyGenerator);

        //先从redis中取得相关数据
        List<RecommendNative> list;
        //如果剩下的不足，就把剩下的全部拿出来，并设置本地人查询为0，表示已经没有本地人了
        if (size < num) {
            list = redisListAdapter.leftIndex(keyGenerator, size - 1, RecommendNative.class);
            redisListAdapter.remove(keyGenerator);
            nativeRecommendContext.setHasSameCity((byte) 0);
        } else {
            list = redisListAdapter.leftIndex(keyGenerator, num - 1, RecommendNative.class);
            redisListAdapter.leftTrim(keyGenerator, num, size);
        }
        nativeRecommendContext.getResult().addAll(list);

        return nativeRecommendContext.getPageSize() == nativeRecommendContext.getResult().size();
    }

    /**
     * 搜索非本地人之前，先判断是否有，如果没有则填充
     *
     * @param nativeRecommendContext
     * @return
     */
    boolean preNearbyInvoke(NativeRecommendContext nativeRecommendContext) {

        KeyGenerator keyGenerator = getKeyGenerator(nativeRecommendContext, RecommendTypeEnum.NEARBY.getName());

        //如果是下拉，需要重新填充
        if (1 == nativeRecommendContext.getActionType()) {
            RecommendNativeParam recommendNativeParam = nativeRecommendContext.getParam(RECOMMEND_NATIVE_PARAM);
            List<RecommendNative> r = recommendNativeMapper.selectByDistance(recommendNativeParam);
            //保存到redis中
            redisListAdapter.remove(keyGenerator);
            if (CollectionUtils.isEmpty(r)) {
                //对不起，真的一滴都没有了
                return false;
            }

            //获取推荐数据之后进行数据找回，降权、拉黑,筛选出拉黑或者降权的用户，不在意这些用户的顺序，所以用并行流
            this.processDataRecall(r, nativeRecommendContext);

            RecommendNative[] recommendNatives = r.toArray(new RecommendNative[0]);
            redisListAdapter.rightPush(keyGenerator, recommendNatives);
            // 当天结束过期
            redisListAdapter.expire(keyGenerator, DateUtils.getCurSeconds());
        } else {
            int size = redisListAdapter.size(keyGenerator).intValue();
            if (size == 0) {
                //对不起，真的一滴都没有了
                return false;
            }
        }

        return true;
    }

    /**
     * 从redis中取出附近的人的列表，并随机插入
     *
     * @param nativeRecommendContext
     */
    boolean doNearbyInvoke(NativeRecommendContext nativeRecommendContext) {

        //还需要取多少条数据
        long num = (long) nativeRecommendContext.getPageSize() - nativeRecommendContext.getResult().size();

        KeyGenerator keyGenerator = getKeyGenerator(nativeRecommendContext, RecommendTypeEnum.NEARBY.getName());

        long size = redisListAdapter.size(keyGenerator);

        //先从redis中取得相关数据
        List<RecommendNative> list;
        //如果剩下的很少了，就把剩下的全部拿出来
        if (size < num) {
            list = redisListAdapter.leftIndex(keyGenerator, size - 1, RecommendNative.class);
            redisListAdapter.remove(keyGenerator);
        } else {
            list = redisListAdapter.leftIndex(keyGenerator, num - 1, RecommendNative.class);
            redisListAdapter.leftTrim(keyGenerator, num, size);
        }

        //取出数据后，如果有同城人，则进行穿插
        if (1 == nativeRecommendContext.getHasSameCity()) {
            List<Integer> indexList = new ArrayList<>();
            for (int i = 0; i < list.size(); i++) {
                //得到穿插的位置
                indexList.add(nextInt(0, nativeRecommendContext.getResult().size()));
            }

            //将位置从大到小排序
            indexList = indexList.stream().sorted((x, y) -> x > y ? 1 : 0).collect(Collectors.toList());

            //开始穿插
            for (int i = 0; i < list.size(); i++) {
                nativeRecommendContext.getResult().add(indexList.get(i), list.get(i));
            }
        } else {
            nativeRecommendContext.getResult().addAll(list);
        }

        return nativeRecommendContext.getPageSize() == nativeRecommendContext.getResult().size();

    }

}
