package com.bxm.localnews.admin.controller.im;

import com.bxm.localnews.admin.constant.RedisConfig;
import com.bxm.localnews.admin.dto.im.ChatRoomMessageDTO;
import com.bxm.localnews.admin.integration.IMIntegrationService;
import com.bxm.localnews.admin.param.ChatRoomQueryParam;
import com.bxm.localnews.admin.param.ImBlockParam;
import com.bxm.localnews.admin.param.ImMessageQueryParam;
import com.bxm.localnews.admin.service.im.IMService;
import com.bxm.localnews.admin.vo.im.HistoryChatBean;
import com.bxm.localnews.admin.vo.im.ImMessageBean;
import com.bxm.localnews.common.constant.RespCode;
import com.bxm.localnews.common.vo.Json;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.vo.PageWarper;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiOperation;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import javax.servlet.http.HttpServletResponse;
import java.util.Date;
import java.util.List;

@Api(tags = "1-59 即时通讯相关接口")
@RestController
@RequestMapping("api/im")
public class ImController {

    private final IMIntegrationService imIntegrationService;

    private final IMService imService;

    private final RedisStringAdapter redisStringAdapter;



    @Autowired
    public ImController(IMIntegrationService imIntegrationService,
                        IMService imService,
                        RedisStringAdapter redisStringAdapter) {
        this.imIntegrationService = imIntegrationService;
        this.imService = imService;
        this.redisStringAdapter = redisStringAdapter;
    }

    @ApiOperation(value = "1-59-1 查询聊天记录（不支持分页）", notes = "可查询双人聊天和单人发起的聊天")
    @GetMapping("query")
    public Json<List<ImMessageBean>> query(ImMessageQueryParam param) {
        boolean validFailed = null == param || (param.getReceiver() == null && param.getSender() == null);
        if (validFailed) {
            return Json.build(RespCode.BAD_REQUEST, "参数必须有发送人或接收人ID，时间也最好提供");
        }
        return Json.build(imService.query(param));
    }

    @ApiOperation(value = "1-59-2 屏蔽用户", notes = "禁止用户在指定的时间段内发起聊天")
    @PostMapping("block")
    public Json<Boolean> block(@RequestBody ImBlockParam param) {
        redisStringAdapter.set(getBlockRedisKey(param.getUserId()), param.getEndTime());
        redisStringAdapter.expire(getBlockRedisKey(param.getUserId()), param.getEndTime());
        return Json.build(imIntegrationService.block(param.getUserId(), param.getEndTime()));
    }

    @ApiOperation(value = "1-59-3 解除屏蔽用户", notes = "允许用户进行聊天")
    @PostMapping("unblock")
    public Json<Boolean> unblock(@RequestParam("userId") Long userId) {
        redisStringAdapter.remove(getBlockRedisKey(userId));
        return Json.build(imIntegrationService.unblock(userId));
    }

    @ApiOperation(value = "1-59-4 查询用户屏蔽结束时间")
    @GetMapping("queryUserBlockEndTime")
    public Json<Date> queryUserBlockEndTime(@RequestParam("userId") Long userId) {
        return Json.build(redisStringAdapter.get(getBlockRedisKey(userId), Date.class));
    }

    @ApiOperation(value = "1-59-5 获取用户token", notes = "用户必须在用户库中存在")
    @ApiImplicitParam(value = "userId", name = "获取token的目标用户ID")
    @GetMapping("token")
    public Json<String> getToken(@RequestParam("userId") Long userId) {
        return Json.ok(imIntegrationService.token(userId));
    }

    @ApiOperation(value = "1-59-6 获取马甲号近期聊天对象信息")
    @ApiImplicitParam(value = "userId", name = "马甲号ID")
    @GetMapping("histories")
    public Json<List<HistoryChatBean>> getHistoryUser(Long userId) {
        return Json.ok(imService.queryHistory(userId));
    }


    @ApiOperation(value = "1-59-7 创建聊天室")
    @PostMapping("createChatRoom")
    public Json<Boolean> createChatRoom(@RequestParam("areaCode") String areaCode) {

        return Json.build(imService.createChatRoom(areaCode));
    }

    @ApiOperation(value = "1-59-8 语音文件下载")
    @GetMapping("getVoiceByContent")
    public void createChatRoom(@RequestParam("id") String id, HttpServletResponse response) {
        imService.getVoiceByContent(id,response);
    }

    @ApiOperation(value = "1-59-9 查询聊天室聊天记录", notes = "")
    @GetMapping("queryChatRoomRecord")
    public Json<PageWarper<ChatRoomMessageDTO>> queryChatRoomRecord(ChatRoomQueryParam param) {
        return Json.build(imService.queryChatRoomRecord(param));
    }

    private KeyGenerator getBlockRedisKey(Long userId) {
        return RedisConfig.IM_BLOCK_LIST.copy().appendKey(userId);
    }

}
