package com.bxm.localnews.admin.service.activity.strategy;

import com.bxm.localnews.admin.config.BizConfigProperties;
import com.bxm.localnews.admin.service.PushMsgIntegService;
import com.bxm.localnews.admin.vo.AdminPushMessage;
import com.bxm.localnews.mq.common.constant.*;
import com.bxm.localnews.mq.common.model.dto.PushMessage;
import com.bxm.localnews.mq.common.model.dto.PushPayloadInfo;
import com.bxm.localnews.mq.common.model.dto.PushReceiveScope;
import com.bxm.newidea.component.tools.StringUtils;
import com.bxm.newidea.component.vo.Message;
import com.google.common.collect.Lists;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.Date;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * @author zhaoyadong 2019/1/17 13:39
 * @desc
 */
@Component
public class PushMessageHandle {
    private static final Logger logger = LoggerFactory.getLogger(PushMessageHandle.class);

    @Autowired
    private PushMsgIntegService pushMsgIntegService;

    @Autowired
    private BizConfigProperties bizConfigProperties;

    /**
     * 推送消息
     * @param adminPushMessage
     */
    public Message pushMessage(AdminPushMessage adminPushMessage) {
        PushMessage pushMessage = generateMessage(adminPushMessage);
        logger.info("推送的消息：{}", pushMessage.toString());
        pushMsgIntegService.pushMsg(pushMessage);

        return Message.build();
    }

    /**
     * 针对单个用户推送
     * @param adminPushMessage 消息
     * @param userId           用户id
     * @return
     */
    public Message pushMessage(AdminPushMessage adminPushMessage, Long userId) {
        adminPushMessage.setId(0L);
        PushMessage pushMessage = generateMessage(adminPushMessage);
        pushMessage.setPushReceiveScope(getPushReceiveScope(userId));
        logger.info("单个用户测试的推送消息为：{}", pushMessage.toString());
        pushMsgIntegService.pushMsg(pushMessage);

        return Message.build();
    }

    /**
     * 生成推送消息的实体
     * @param adminPushMessage
     * @return
     */
    public PushMessage generateMessage(AdminPushMessage adminPushMessage) {
        PushMessage pushMessage = PushMessage.build().setTitle(adminPushMessage.getTitle())
                .setContent(adminPushMessage.getContent()).setMute(byteToBoolean(adminPushMessage.getMute()));

        pushMessage.setType(getTemplateType(adminPushMessage.getPushType()));
        pushMessage.setFloatNotify(byteToBoolean(adminPushMessage.getFloatNotify()));
        pushMessage.setPushFoldType(getPushFold(adminPushMessage.getUnfoldType()));
        pushMessage.setPeriodTime(adminPushMessage.getPeriodTime());
        pushMessage.setShock(byteToBoolean(adminPushMessage.getVibrate()));
        pushMessage.setActionType(adminPushMessage.getActionType());

        if (StringUtils.isNotEmpty(adminPushMessage.getNotifyImgUrl())) {
            pushMessage.setNotifyImageUrl(adminPushMessage.getNotifyImgUrl());
        }

        if (StringUtils.isNotEmpty(adminPushMessage.getNotifyText())) {
            pushMessage.setNotifyText(adminPushMessage.getNotifyText());
        }

        pushMessage.setPayloadInfo(getPushInfo(adminPushMessage));
        pushMessage.setPushReceiveScope(getPushReceiveScope(adminPushMessage));

        return pushMessage;
    }

    /**
     * byte值转换为boolean值
     * @param value
     * @return
     */
    private Boolean byteToBoolean(Byte value) {
        return value == 0 ? false : true;
    }

    /**
     * 通知方式
     * @return
     */
    private PushFoldTypeEnum getPushFold(String unfoldType) {
        return PushFoldTypeEnum.valueOf(unfoldType);
    }

    /**
     * 推送方式
     * @param pushType
     * @return
     */
    private TemplateTypeEnum getTemplateType(String pushType) {

        return TemplateTypeEnum.valueOf(pushType);
    }

    /**
     * 组装参数
     * @param adminPushMessage
     * @return
     */
    private PushPayloadInfo getPushInfo(AdminPushMessage adminPushMessage) {
        Date now = new Date();
        PushPayloadInfo payloadInfo = PushPayloadInfo.build().setTitle(adminPushMessage.getTitle())
                .setContent(adminPushMessage.getContent())
                .setCreateTime(now.getTime());
        payloadInfo.setMsgId(adminPushMessage.getId());
        payloadInfo.addExtend("title", adminPushMessage.getTitle());
        if (ActionTypeEnum.OPEN_NEWS.getType().equals(adminPushMessage.getActionType())) {
            payloadInfo.setType(PushMessageEnum.NEWS.getType());
            payloadInfo.addExtend("newsId", adminPushMessage.getActionParams());
            payloadInfo.addExtend("type", 1);
        }

        if (ActionTypeEnum.OPEN_URL.getType().equals(adminPushMessage.getActionType())) {
            payloadInfo.setType(PushMessageEnum.OPEN_URL.getType());
            payloadInfo.addExtend("url", adminPushMessage.getActionParams());
        }

        if (ActionTypeEnum.VIP_NOT_ACTIVATION.getType().equals(adminPushMessage.getActionType())) {
            payloadInfo.setType(PushMessageEnum.VIP_NOT_ACTIVATION.getType());

            payloadInfo.addExtend("url", bizConfigProperties.getH5ServerHost() + "/openVIP.html?isLogin={isLogin}&userId={userId}&code={areaCode}" +
                    "&activate=0&cityName={areaName}&longitude={lng}&latitude={lat}&page_src=push");
        }

        if (ActionTypeEnum.VIP_CONTINUE_ACTIVATION.getType().equals(adminPushMessage.getActionType())) {
            payloadInfo.setType(PushMessageEnum.VIP_CONTINUE_ACTIVATION.getType());
            payloadInfo.addExtend("url", bizConfigProperties.getH5ServerHost() + "/inviteFriendsVIP.html?userId={userId}&code={areaCode}" +
                    "&cityName={areaName}&activate=0&page_src=push");
        }

        if (ActionTypeEnum.VIP_ACTIVATION.getType().equals(adminPushMessage.getActionType())) {
            payloadInfo.setType(PushMessageEnum.VIP_ACTIVATION.getType());
            payloadInfo.addExtend("url", bizConfigProperties.getH5ServerHost() + "/shareCard.html?userId={userId}&code={areaCode}" +
                    "&cityName={areaName}&longitude={lng}&latitude={lat}&page_src=push");
        }

        return payloadInfo;
    }

    /**
     * 推送规则设置
     * @return
     */
    private PushReceiveScope getPushReceiveScope(AdminPushMessage adminPushMessage) {
        //设置平台
        List<PlatformTypeEnum> platformTypeEnumList = Lists.newArrayList();
        if (StringUtils.isNotEmpty(adminPushMessage.getPushPlatform())) {
            String[] platforms = adminPushMessage.getPushPlatform().split(",");
            for (String platform : platforms) {
                platformTypeEnumList.add(PlatformTypeEnum.valueOf(platform));
            }
        }

        //如果是vip推送类消息
        if (ActionTypeEnum.VIP_NOT_ACTIVATION.getType().equals(adminPushMessage.getActionType())
                || ActionTypeEnum.VIP_CONTINUE_ACTIVATION.getType().equals(adminPushMessage.getActionType())
                || ActionTypeEnum.VIP_ACTIVATION.getType().equals(adminPushMessage.getActionType())) {

            return buildPushReceiveScope(adminPushMessage).setPlatformTypes(platformTypeEnumList);
        }

        //设置目标用户
        if (adminPushMessage.getPushTarget() == 1) {
            String[] userIds = adminPushMessage.getTargetUserid().split(",");
            return PushReceiveScope.pushGroup(Stream.of(userIds).map(Long::new).collect(Collectors.toList()))
                    .setPlatformTypes(platformTypeEnumList);
        }

        if (adminPushMessage.getPushTarget() == 2) {
            String[] areas = adminPushMessage.getAreaScope().split(",");
            return PushReceiveScope.pushRule(PushReceiverRuleEnum.AREA).addRuleParam(ReceiverRuleParamConstant.AREA_CODES,
                    Stream.of(areas).collect(Collectors.toList()))
                    .setPlatformTypes(platformTypeEnumList);
        }

        if (adminPushMessage.getPushTarget() == 3) {
            return PushReceiveScope.pushRule(PushReceiverRuleEnum.ALL).setPlatformTypes(platformTypeEnumList);
        }

        return null;
    }

    /**
     * 推送单个用户
     * @param userId
     * @return
     */
    private PushReceiveScope getPushReceiveScope(Long userId) {
        return PushReceiveScope.pushSignle(userId);
    }

    /**
     * 根据参数构建对应的消息接受范围
     * @param adminPushMessage
     * @return
     */
    private PushReceiveScope buildPushReceiveScope(AdminPushMessage adminPushMessage) {
        if (adminPushMessage.getPushTarget() == 1) {
            String[] userIds = adminPushMessage.getTargetUserid().split(",");
            return PushReceiveScope.pushRule(PushReceiverRuleEnum.VIP_USER)
                    .addRuleParam(ReceiverRuleParamConstant.VIP, adminPushMessage.getActionType().byteValue())
                    .addRuleParam(ReceiverRuleParamConstant.USER_IDS, Stream.of(userIds).map(Long::new)
                            .collect(Collectors.toList()));
        }

        if (adminPushMessage.getPushTarget() == 2) {
            String[] areas = adminPushMessage.getAreaScope().split(",");
            return PushReceiveScope.pushRule(PushReceiverRuleEnum.VIP_AREA)
                    .addRuleParam(ReceiverRuleParamConstant.VIP, adminPushMessage.getActionType().byteValue())
                    .addRuleParam(ReceiverRuleParamConstant.AREA_CODES,
                            Stream.of(areas).collect(Collectors.toList()));
        }

        if (adminPushMessage.getPushTarget() == 3) {
            return PushReceiveScope.pushRule(PushReceiverRuleEnum.VIP_AREA)
                    .addRuleParam(ReceiverRuleParamConstant.VIP, adminPushMessage.getActionType());
        }
        return null;
    }
}
