package com.bxm.localnews.admin.service.base.impl;

import java.math.BigDecimal;
import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

import com.bxm.localnews.admin.constant.RedisConfig;
import com.bxm.localnews.admin.domain.AreaDivisionMapper;
import com.bxm.localnews.admin.domain.AreaInfoMapper;
import com.bxm.localnews.admin.domain.LocationMapper;
import com.bxm.localnews.admin.domain.LocationRelationMapper;
import com.bxm.localnews.admin.dto.AreaInfoDTO;
import com.bxm.localnews.admin.dto.LocationInfoDTO;
import com.bxm.localnews.admin.param.AreaInfoParam;
import com.bxm.localnews.admin.param.AreaInfoQueryParam;
import com.bxm.localnews.admin.service.base.AreaInfoService;
import com.bxm.localnews.admin.vo.AreaDivision;
import com.bxm.localnews.admin.vo.AreaInfo;
import com.bxm.localnews.admin.vo.Location;
import com.bxm.localnews.admin.vo.LocationRelationBean;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.service.BaseService;
import com.bxm.newidea.component.tools.StringUtils;
import com.bxm.newidea.component.vo.Message;
import com.bxm.newidea.component.vo.PageWarper;
import com.google.common.collect.Lists;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

@Service
public class AreaInfoServiceImpl extends BaseService implements AreaInfoService {

    private final AreaInfoMapper areaInfoMapper;

    private final AreaDivisionMapper areaDivisionMapper;

    private final RedisStringAdapter redisStringAdapter;

    private final LocationRelationMapper locationRelationMapper;

    private final LocationMapper locationMapper;

    @Autowired
    public AreaInfoServiceImpl(AreaInfoMapper areaInfoMapper, AreaDivisionMapper areaDivisionMapper,
                               RedisStringAdapter redisStringAdapter, LocationRelationMapper locationRelationMapper,
                               LocationMapper locationMapper) {
        this.areaInfoMapper = areaInfoMapper;
        this.areaDivisionMapper = areaDivisionMapper;
        this.redisStringAdapter = redisStringAdapter;
        this.locationRelationMapper = locationRelationMapper;
        this.locationMapper = locationMapper;
    }

    @Override
    public PageWarper<AreaInfo> queryPages(AreaInfoQueryParam areaInfoParam) {
        PageWarper<AreaInfo> pageWarper = new PageWarper<>(areaInfoMapper.queryByPageSize(areaInfoParam));
        pageWarper.getList().forEach(areaInfo -> {
            Location location = locationMapper.selectByCode(areaInfo.getCode());
            if (location != null) {
                areaInfo.setEnableVip(location.getEnableVip());
                areaInfo.setEnablePaidPromote(location.getEnablePaidPromote());
                areaInfo.setEnableCommunityContent(location.getEnableCommunityContent());
                areaInfo.setEnableIndexPublish(location.getEnableIndexPublish());
                areaInfo.setEnableCustomerService(location.getEnableCustomerService());
            }
        });
        return pageWarper;
    }

    @Override
    public Message saveAreaInfo(AreaInfoParam areaInfoParam) {
        AreaInfo areaInfo = areaInfoMapper.selectByCode(areaInfoParam.getCode());
        if (null != areaInfo) {
            return Message.build(false, "请勿重复添加");
        }
        AreaInfo areaInfoEntity = convertAreaInfo(areaInfoParam);
        int count = areaInfoMapper.insertSelective(areaInfoEntity);
        saveLocation(areaInfoParam);

        if (count == 1) {
            List<AreaInfo> areaInfoList = areaInfoMapper.queryByPageSize(null);
            redisStringAdapter.set(RedisConfig.AREA_INFO, areaInfoList);
        }

        return Message.build(count);
    }

    private AreaInfo convertAreaInfo(AreaInfoParam areaInfoParam) {
        AreaInfo areaInfo = new AreaInfo();
        areaInfo.setName(areaInfoParam.getName());
        areaInfo.setCode(areaInfoParam.getCode());
        areaInfo.setCoverRate(areaInfoParam.getCoverRate());
        areaInfo.setDescription(areaInfoParam.getDescription());
        areaInfo.setPopulation(areaInfoParam.getPopulation());
        areaInfo.setTransmissionRate(areaInfoParam.getTransmissionRate());


        if (areaInfoParam.getId() != null) {
            areaInfo.setId(areaInfoParam.getId());
        } else {
            areaInfo.setId(nextSequence());
        }

        return areaInfo;
    }

    /**
     * 保存location（定位表）信息，同时更新location_relation（定位关联信息）
     *
     * @param areaInfo
     */
    private void saveLocation(AreaInfoParam areaInfo) {
        Location location = new Location();
        location.setCode(areaInfo.getCode());
        location.setEnableVip(areaInfo.getEnableVip());
        location.setInvitedCount(areaInfo.getInvitedCount());
        location.setReductionAmount(areaInfo.getValue());
        location.setEnablePaidPromote(areaInfo.getEnablePaidPromote());
        if (StringUtils.isNotEmpty(areaInfo.getNewRedPacket())) {
            location.setNewRedPacket(new BigDecimal(areaInfo.getNewRedPacket()));
        }
        if (StringUtils.isNotEmpty(areaInfo.getPromoteBounty())) {
            location.setPromoteBounty(new BigDecimal(areaInfo.getPromoteBounty()));
        }

        locationMapper.updateByCodeSelective(location);
        if (areaInfo.getType() == null) {
            areaInfo.setType(0);
        }
        if (areaInfo.getType() == 1) {
            areaInfo.getShareWechatSeg().forEach(seg -> {
                if (seg.getId() == null) {
                    seg.setId(nextId());
                }
                seg.setType((byte) 1);
                seg.setCreateTime(new Date());
                seg.setLocationCode(areaInfo.getCode());
            });

            areaInfo.getShareMomentSeg().forEach(momentSeg -> {
                if (momentSeg.getId() == null) {
                    momentSeg.setId(nextId());
                }
                momentSeg.setType((byte) 2);
                momentSeg.setCreateTime(new Date());
                momentSeg.setLocationCode(areaInfo.getCode());
            });

            areaInfo.getShareWechatImgSeg().forEach(seg -> {
                if (seg.getId() == null) {
                    seg.setId(nextId());
                }
                seg.setType((byte) 3);
                seg.setCreateTime(new Date());
                seg.setLocationCode(areaInfo.getCode());
            });

            areaInfo.getShareMomentImgSeg().forEach(momentSeg -> {
                if (momentSeg.getId() == null) {
                    momentSeg.setId(nextId());
                }
                momentSeg.setType((byte) 4);
                momentSeg.setCreateTime(new Date());
                momentSeg.setLocationCode(areaInfo.getCode());
            });
        }

        if (areaInfo.getType() == 2) {
            areaInfo.getShareMiniAppWechatSeg().forEach(momentSeg -> {
                if (momentSeg.getId() == null) {
                    momentSeg.setId(nextId());
                }
                momentSeg.setType((byte) 5);
                momentSeg.setCreateTime(new Date());
                momentSeg.setLocationCode(areaInfo.getCode());
            });

            areaInfo.getShareMiniAppWechatImgSeg().forEach(momentSeg -> {
                if (momentSeg.getId() == null) {
                    momentSeg.setId(nextId());
                }
                momentSeg.setType((byte) 6);
                momentSeg.setCreateTime(new Date());
                momentSeg.setLocationCode(areaInfo.getCode());
            });

            areaInfo.getShareMiniAppMomentSeg().forEach(momentSeg -> {
                if (momentSeg.getId() == null) {
                    momentSeg.setId(nextId());
                }
                momentSeg.setType((byte) 7);
                momentSeg.setCreateTime(new Date());
                momentSeg.setLocationCode(areaInfo.getCode());
            });

            areaInfo.getShareMiniAppMomentImgSeg().forEach(momentSeg -> {
                if (momentSeg.getId() == null) {
                    momentSeg.setId(nextId());
                }
                momentSeg.setType((byte) 8);
                momentSeg.setCreateTime(new Date());
                momentSeg.setLocationCode(areaInfo.getCode());
            });

            areaInfo.getRecallMiniAppWechatImgSeg().forEach(momentSeg -> {
                if (momentSeg.getId() == null) {
                    momentSeg.setId(nextId());
                }
                momentSeg.setType((byte) 9);
                momentSeg.setCreateTime(new Date());
                momentSeg.setLocationCode(areaInfo.getCode());
            });

            areaInfo.getRecallMiniAppMomentImgSeg().forEach(momentSeg -> {
                if (momentSeg.getId() == null) {
                    momentSeg.setId(nextId());
                }
                momentSeg.setType((byte) 10);
                momentSeg.setCreateTime(new Date());
                momentSeg.setLocationCode(areaInfo.getCode());
            });
        }

        List<LocationRelationBean> relationBeans = locationRelationMapper.selectByCodeAndType(areaInfo.getCode(),
                areaInfo.getType());
        List<LocationRelationBean> paramRelationBeans = Lists.newArrayList();
        if (areaInfo.getType() == 1) {
            paramRelationBeans.addAll(areaInfo.getShareWechatSeg());
            paramRelationBeans.addAll(areaInfo.getShareWechatImgSeg());
            paramRelationBeans.addAll(areaInfo.getShareMomentSeg());
            paramRelationBeans.addAll(areaInfo.getShareMomentImgSeg());
        }

        if (areaInfo.getType() == 2) {
            paramRelationBeans.addAll(areaInfo.getShareMiniAppWechatSeg());
            paramRelationBeans.addAll(areaInfo.getShareMiniAppWechatImgSeg());
            paramRelationBeans.addAll(areaInfo.getShareMiniAppMomentSeg());
            paramRelationBeans.addAll(areaInfo.getShareMiniAppMomentImgSeg());
            paramRelationBeans.addAll(areaInfo.getRecallMiniAppWechatImgSeg());
            paramRelationBeans.addAll(areaInfo.getRecallMiniAppMomentImgSeg());

        }

        for (LocationRelationBean entity : paramRelationBeans) {
            if (StringUtils.isBlank(entity.getContent())) {
                locationRelationMapper.deleteById(entity.getId());
                continue;
            }
            locationRelationMapper.insert(entity);
        }

        for (LocationRelationBean entity : relationBeans) {
            if (paramRelationBeans.stream().anyMatch(relation -> Objects.equals(entity.getId(), relation.getId()))) {
                continue;
            }

            locationRelationMapper.deleteById(entity.getId());
        }
    }

    @Override
    public int updateAreaInfo(AreaInfoParam areaInfoParam) {
        AreaInfo areaInfo = convertAreaInfo(areaInfoParam);

        int count = areaInfoMapper.updateByPrimaryKeySelective(areaInfo);
        saveLocation(areaInfoParam);

        if (count == 1) {
            List<AreaInfo> areaInfoList = areaInfoMapper.queryByPageSize(null);
            redisStringAdapter.set(RedisConfig.AREA_INFO, areaInfoList);
        }
        return count;
    }

    @Override
    public int delAreaInfo(String code) {
        int count = areaInfoMapper.deleteByCode(code);
        locationRelationMapper.delete(code);
        if (count == 1) {
            List<AreaInfo> areaInfoList = areaInfoMapper.queryByPageSize(null);
            redisStringAdapter.set(RedisConfig.AREA_INFO, areaInfoList);
        }

        return count;
    }

    @Override
    public AreaInfoDTO getAreaInfo(String code) {
        AreaInfo areaInfo = areaInfoMapper.selectByCode(code);
        AreaDivision areaDivision = areaDivisionMapper.selectByCode(areaInfo.getCode());
        String parentCode = areaDivision.getParentCode();
        if (null != parentCode) {
            AreaDivision areaDivisionUp = areaDivisionMapper.selectByCode(parentCode);
            if (null != areaDivisionUp.getParentCode()) {
                parentCode = areaDivisionUp.getParentCode();
            }
        }
        String parentName = areaDivisionMapper.selectByCode(parentCode).getName();

        AreaInfoDTO areaInfoDTO = new AreaInfoDTO();
        areaInfoDTO.setId(areaInfo.getId());
        areaInfoDTO.setProvinceCode(parentCode);
        areaInfoDTO.setName(areaInfo.getName());
        areaInfoDTO.setProvinceName(parentName);
        areaInfoDTO.setCode(areaInfo.getCode());
        areaInfoDTO.setCoverRate(areaInfo.getCoverRate());
        areaInfoDTO.setTransmissionRate(areaInfo.getTransmissionRate());
        areaInfoDTO.setPopulation(areaInfo.getPopulation());
        areaInfoDTO.setDescription(areaInfo.getDescription());

        Location location = locationMapper.selectByCode(code);
        areaInfoDTO.setEnableVip(location.getEnableVip());
        areaInfoDTO.setInvitedCount(location.getInvitedCount());
        areaInfoDTO.setValue(location.getReductionAmount());

        areaInfoDTO.setEnablePaidPromote(location.getEnablePaidPromote());
        areaInfoDTO.setNewRedPacket(location.getNewRedPacket());
        areaInfoDTO.setPromoteBounty(location.getPromoteBounty());

        List<LocationRelationBean> relationBeans = locationRelationMapper.selectByCode(code);
        areaInfoDTO.setShareWechatSeg(relationBeans.stream().filter(seg -> seg.getType() == 1).collect(Collectors.toList()));
        areaInfoDTO.setShareMomentSeg(relationBeans.stream().filter(seg -> seg.getType() == 2).collect(Collectors.toList()));
        areaInfoDTO.setShareWechatImgSeg(relationBeans.stream().filter(seg -> seg.getType() == 3).collect(Collectors.toList()));
        areaInfoDTO.setShareMomentImgSeg(relationBeans.stream().filter(seg -> seg.getType() == 4).collect(Collectors.toList()));
        areaInfoDTO.setShareMiniAppWechatSeg(relationBeans.stream().filter(seg -> seg.getType() == 5).collect(Collectors.toList()));
        areaInfoDTO.setShareMiniAppWechatImgSeg(relationBeans.stream().filter(seg -> seg.getType() == 6).collect(Collectors.toList()));
        areaInfoDTO.setShareMiniAppMomentSeg(relationBeans.stream().filter(seg -> seg.getType() == 7).collect(Collectors.toList()));
        areaInfoDTO.setShareMiniAppMomentImgSeg(relationBeans.stream().filter(seg -> seg.getType() == 8).collect(Collectors.toList()));
        areaInfoDTO.setRecallMiniAppWechatImgSeg(relationBeans.stream().filter(seg -> seg.getType() == 9).collect(Collectors.toList()));
        areaInfoDTO.setRecallMiniAppMomentImgSeg(relationBeans.stream().filter(seg -> seg.getType() == 10).collect(Collectors.toList()));
        return areaInfoDTO;
    }

    @Override
    public int updateCommunityContent(AreaInfoParam areaInfo) {
        Location location = new Location();
        location.setCode(areaInfo.getCode());
        //更新社区内容
        if (null != areaInfo.getEnableCommunityContent()) {
            location.setEnableCommunityContent(areaInfo.getEnableCommunityContent());
        }

        return locationMapper.updateByCodeSelective(location);
    }

    @Override
    public int updateCustomerService(AreaInfoParam areaInfoParam) {
        Location location = new Location();
        location.setCode(areaInfoParam.getCode());

        //更新客服小助手
        if (null != areaInfoParam.getEnableCustomerService()) {
            location.setEnableCustomerService(areaInfoParam.getEnableCustomerService());
            location.setCustomerServiceNumber(areaInfoParam.getCustomerServiceNumber());
            location.setCustomerServiceQrCode(areaInfoParam.getCustomerServiceQrCode());
        }
        return locationMapper.updateByCodeSelective(location);

    }

    @Override
    public int updateIndexPublish(AreaInfoParam areaInfo) {
        Location location = new Location();
        location.setCode(areaInfo.getCode());

        //更新首页发帖
        if (null != areaInfo.getEnableIndexPublish()) {
            location.setEnableIndexPublish(areaInfo.getEnableIndexPublish());
        }
        return locationMapper.updateByCodeSelective(location);

    }

    @Override
    public LocationInfoDTO getLocationInfo(String code) {
        Location location = locationMapper.selectByCode(code);
        LocationInfoDTO locationInfoDTO = new LocationInfoDTO();

        if (null == location) {
            logger.error("查询location信息为空,查询code为:{}", code);
            return null;
        }

        //参数转换
        locationInfoDTO.setEnableCommunityContent(location.getEnableCommunityContent());
        locationInfoDTO.setEnableCustomerService(location.getEnableCustomerService());
        locationInfoDTO.setEnableIndexPublish(location.getEnableIndexPublish());
        locationInfoDTO.setCustomerServiceNumber(location.getCustomerServiceNumber());
        locationInfoDTO.setCustomerServiceQrCode(location.getCustomerServiceQrCode());
        locationInfoDTO.setCode(location.getCode());
        locationInfoDTO.setName(location.getName());

        return locationInfoDTO;
    }
}
