package com.bxm.localnews.admin.config;

import java.util.Map;

import com.bxm.localnews.admin.config.jwt.JwtAuthenticationEntryPoint;
import com.bxm.localnews.admin.config.jwt.JwtAuthenticationTokenFilter;

import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.http.HttpMethod;
import org.springframework.security.access.expression.SecurityExpressionHandler;
import org.springframework.security.authentication.AuthenticationManager;
import org.springframework.security.config.annotation.authentication.builders.AuthenticationManagerBuilder;
import org.springframework.security.config.annotation.method.configuration.EnableGlobalMethodSecurity;
import org.springframework.security.config.annotation.web.builders.HttpSecurity;
import org.springframework.security.config.annotation.web.builders.WebSecurity;
import org.springframework.security.config.annotation.web.configuration.EnableWebSecurity;
import org.springframework.security.config.annotation.web.configuration.WebSecurityConfigurerAdapter;
import org.springframework.security.config.http.SessionCreationPolicy;
import org.springframework.security.core.userdetails.UserDetailsService;
import org.springframework.security.crypto.password.PasswordEncoder;
import org.springframework.security.web.FilterInvocation;
import org.springframework.security.web.access.intercept.FilterSecurityInterceptor;
import org.springframework.security.web.authentication.UsernamePasswordAuthenticationFilter;

/**
 * 权限管理配置,启用无状态的基于jwt的权限控制
 *
 * @author liujia 2018/8/3 18:19
 */
@Configuration
@EnableWebSecurity
@EnableGlobalMethodSecurity(prePostEnabled = true)
public class SecurityConfiguration extends WebSecurityConfigurerAdapter {

    @Autowired
    private UserDetailsService userDetailsService;

    @Autowired
    private PasswordEncoder passwordEncoder;

    @Autowired
    private AuthorizationProperties authorizationProperties;

    private HttpSecurity httpSecurity;

    @Autowired
    public void configureAuthentication(AuthenticationManagerBuilder authenticationManagerBuilder) throws Exception {
        authenticationManagerBuilder
                // 设置UserDetailsService
                .userDetailsService(this.userDetailsService)
                //
                .passwordEncoder(this.passwordEncoder);
    }

    @Override
    public void configure(WebSecurity web) throws Exception {
        SecurityExpressionHandler<FilterInvocation> handler = web.getExpressionHandler();
        AuthenticationManager authenticationManager = authenticationManager();
        //初始化完成后重新设置interceptor
        web.postBuildAction(() -> {
            FilterSecurityInterceptor interceptor = httpSecurity.getSharedObject(FilterSecurityInterceptor.class);
            interceptor.setAuthenticationManager(authenticationManager);
        });
    }

    @Bean
    public JwtAuthenticationEntryPoint authenticationEntryPoint() {
        return new JwtAuthenticationEntryPoint("/api/admin/toLoginNew");
    }

    @Override
    protected void configure(HttpSecurity http) throws Exception {
        //禁用csrf
        http.csrf().disable()
                .exceptionHandling()
                .accessDeniedHandler(accessDeniedHandler())
                .authenticationEntryPoint(authenticationEntryPoint())
                .and()
                //不创建SESSION
                .sessionManagement().sessionCreationPolicy(SessionCreationPolicy.STATELESS).and()
                .authorizeRequests()
                //允许所有的探测请求
                .antMatchers(HttpMethod.OPTIONS, "/**").permitAll()
                //druid数据源监控允许访问
                .antMatchers("/druid/**").permitAll()
                //运行访问授权控制相关路径
                .antMatchers("/auth/**").permitAll()
                //swagger api访问路径
                .antMatchers("/v2/api-docs").permitAll()
                .antMatchers("/v2/api-docs-ext").permitAll()
                .antMatchers("/swagger-ui.html").permitAll()
                .antMatchers("/doc.html").permitAll()
                .antMatchers("/webjars/**").permitAll()
                //rest API
                .antMatchers("/swagger/**").permitAll()
                .antMatchers("/app/**").permitAll()
                //swagger静态资源
                .antMatchers("/swagger-resources/**").permitAll()
                .antMatchers("/ueditor/**").permitAll()
                .antMatchers("/api/admin/toLoginNew").permitAll()
                .antMatchers("/api/admin/vcode").permitAll()
                .antMatchers("/api/admin/logout").permitAll()
                .antMatchers("/api/admin/push/code").permitAll()
                .antMatchers("/api/admin/push/code/bind").permitAll()
                .antMatchers("/api/admin/push/code/exist").permitAll()
                .antMatchers("/api/admin/marketOrder/exportExcel").permitAll()
                .antMatchers("/api/admin/goldFlow/exportExcel").permitAll()
                //剩下的全部请求都需要进行鉴权
                .anyRequest().authenticated();

        //路径权限配置
        if (!authorizationProperties.getRoleUrl().isEmpty()) {
            for (Map.Entry<String, String> entry : authorizationProperties.getRoleUrl().entrySet()) {
                if (entry.getValue().contains(",")) {
                    String[] str = StringUtils.split(entry.getValue(), ",");
                    for (String s : str) {
                        http.authorizeRequests().antMatchers(entry.getKey()).hasRole(s);
                    }
                } else {
                    http.authorizeRequests().antMatchers(entry.getKey()).hasRole(entry.getValue());
                }
            }
        }

        // 添加JWT filter
        http.addFilterBefore(authenticationTokenFilterBean(), UsernamePasswordAuthenticationFilter.class);

        // 禁用缓存
        http.headers().cacheControl();

        this.httpSecurity = http;
    }

    @Bean
    public AuthorizationProperties getAuthorizationProperties() {
        return new AuthorizationProperties();
    }

    @Bean
    public JwtAuthenticationTokenFilter authenticationTokenFilterBean() {
        return new JwtAuthenticationTokenFilter();
    }

    @Bean
    public CustomAccessDeniedHandler accessDeniedHandler() {
        return new CustomAccessDeniedHandler();
    }

    @Bean
    @Override
    public AuthenticationManager authenticationManager() throws Exception {
        return super.authenticationManager();
    }
}
