package com.bxm.localnews.admin.config.jwt;

import com.bxm.localnews.admin.constant.SecurityConstant;
import com.bxm.localnews.admin.service.security.JwtTokenService;
import com.bxm.newidea.component.log.LogMarker;
import com.bxm.newidea.component.util.WebUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpMethod;
import org.springframework.security.authentication.UsernamePasswordAuthenticationToken;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.security.core.userdetails.UserDetails;
import org.springframework.security.core.userdetails.UserDetailsService;
import org.springframework.security.web.authentication.WebAuthenticationDetailsSource;
import org.springframework.web.filter.OncePerRequestFilter;

import javax.servlet.FilterChain;
import javax.servlet.ReadListener;
import javax.servlet.ServletException;
import javax.servlet.ServletInputStream;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletRequestWrapper;
import javax.servlet.http.HttpServletResponse;
import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStreamReader;

/**
 * 每次请求解析并处理token，如果不存在则认为无权访问需要鉴权的资源
 */
public class JwtAuthenticationTokenFilter extends OncePerRequestFilter {

    private static final Logger LOGGER = LoggerFactory.getLogger(JwtAuthenticationTokenFilter.class);

    @Autowired
    private UserDetailsService userDetailsService;

    @Autowired
    private JwtTokenService jwtTokenService;

    @Override
    protected void doFilterInternal(HttpServletRequest request, HttpServletResponse response,
                                    FilterChain chain) throws ServletException, IOException {
        //修改request，使得可以读取多次requestbody
        ContentCachingRequestWrapper cacheRequest = new ContentCachingRequestWrapper(request);
        //从请求和cookie中获取token
        String authToken = cacheRequest.getHeader(SecurityConstant.HEAD_KEY);
//        if (authToken == null) {
//            Cookie[] cookies = request.getCookies();
//            if (cookies != null && cookies.length > 0) {
//                for (Cookie cookie : cookies) {
//                    if (StringUtils.equals(cookie.getName(), SecurityConstant.HEAD_KEY)) {
//                        authToken = cookie.getValue();
//                        break;
//                    }
//                }
//            }
//        }

        if (StringUtils.isNotBlank(authToken)) {
            String account = jwtTokenService.getUsernameFromToken(authToken);

            //当前上下文中没有用户信息则获取后放到上下文中
            if (account != null && SecurityContextHolder.getContext().getAuthentication() == null) {
                //从缓存中加载用户信息
                UserDetails userDetails = this.userDetailsService.loadUserByUsername(account);

                accessLog(cacheRequest, userDetails);

                //校验token是否可用
                if (jwtTokenService.validateToken(authToken, userDetails)) {
                    UsernamePasswordAuthenticationToken authentication = new UsernamePasswordAuthenticationToken(userDetails, null, userDetails.getAuthorities());
                    authentication.setDetails(new WebAuthenticationDetailsSource().buildDetails(cacheRequest));
                    SecurityContextHolder.getContext().setAuthentication(authentication);

                    //如果token即将过期则颁发新的token设置到response中
                    String newToken = jwtTokenService.renewToken(authToken);
                    if (null != newToken) {
                        response.addHeader(SecurityConstant.RESPONSE_TOKEN_HEAD, newToken);
                        response.addHeader("Access-Control-Expose-Headers", SecurityConstant.RESPONSE_TOKEN_HEAD);
                    }
                }
            }
        } else {
            accessLog(cacheRequest, null);
        }

        chain.doFilter(cacheRequest, response);
    }

    private void accessLog(ContentCachingRequestWrapper request, UserDetails userDetails) {
        String requestIp = WebUtils.getIpAddr(request);
        String url = request.getRequestURI();
        String userName = null == userDetails ? null : userDetails.getUsername();
        String params = WebUtils.joinRequestParam(request, "|");
        String requestMethod = request.getMethod();

        if (HttpMethod.OPTIONS.matches(requestMethod)) {
            LOGGER.debug("url:[{}],requestMethod:[{}]", url, requestMethod);
            return;
        }

        //slb负载均衡会在额定的时间间隔内请求,过滤日志
        if (!"/app/heartbeat".equals(url)) {
            byte[] body = request.getBody();

            String reqeustBody = StringUtils.EMPTY;
            if (null != body && body.length > 0) {
                reqeustBody = new String(body);
            }

            LOGGER.info(LogMarker.OPERATION, "url:[{}],requestMethod:[{}],user:[{}],requestIp:[{}],params:[{}],requestBody:[{}]",
                    url,
                    requestMethod,
                    userName,
                    requestIp,
                    params,
                    reqeustBody);
        }
    }

    private class ContentCachingRequestWrapper extends HttpServletRequestWrapper {

        private byte[] body;

        private BufferedReader reader;

        private ServletInputStream inputStream;

        private ContentCachingRequestWrapper(HttpServletRequest request) throws IOException {
            super(request);
            loadBody(request);
        }

        private void loadBody(HttpServletRequest request) throws IOException {
            body = IOUtils.toByteArray(request.getInputStream());
            inputStream = new RequestCachingInputStream(body);
        }

        private byte[] getBody() {
            return body;
        }

        @Override
        public ServletInputStream getInputStream() throws IOException {
            if (inputStream != null) {
                return inputStream;
            }
            return super.getInputStream();
        }

        @Override
        public BufferedReader getReader() throws IOException {
            if (reader == null) {
                reader = new BufferedReader(new InputStreamReader(inputStream, getCharacterEncoding()));
            }
            return reader;
        }

        private class RequestCachingInputStream extends ServletInputStream {

            private final ByteArrayInputStream inputStream;

            private RequestCachingInputStream(byte[] bytes) {
                inputStream = new ByteArrayInputStream(bytes);
            }

            @Override
            public int read() throws IOException {
                return inputStream.read();
            }

            @Override
            public boolean isFinished() {
                return inputStream.available() == 0;
            }

            @Override
            public boolean isReady() {
                return true;
            }

            @Override
            public void setReadListener(ReadListener readlistener) {
            }

        }

    }
}
