package com.bxm.localnews.admin.service.base.impl;

import com.bxm.localnews.admin.constant.RedisConfig;
import com.bxm.localnews.admin.constant.TooltipStrategyEnum;
import com.bxm.localnews.admin.domain.AdminToolsLocationMapper;
import com.bxm.localnews.admin.domain.AdminToolsMapper;
import com.bxm.localnews.admin.dto.ToolsDTO;
import com.bxm.localnews.admin.dto.TooltipStrategyDTO;
import com.bxm.localnews.admin.param.ToolsInfoParam;
import com.bxm.localnews.admin.param.ToolsParam;
import com.bxm.localnews.admin.service.base.AdminToolsService;
import com.bxm.localnews.admin.vo.Tools;
import com.bxm.localnews.admin.vo.ToolsLocation;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.tools.StringUtils;
import com.bxm.newidea.component.uuid.SequenceCreater;
import com.bxm.newidea.component.vo.PageWarper;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.List;

/**
 * @author zhaoyadong 2018/10/23 13:58
 * @desc
 */
@Service
public class AdminToolsServiceImpl implements AdminToolsService {

    private AdminToolsMapper adminToolsMapper;

    private AdminToolsLocationMapper adminToolsLocationMapper;

    private RedisHashMapAdapter redisHashMapAdapter;

    private SequenceCreater sequenceCreater;

    @Autowired
    public AdminToolsServiceImpl(AdminToolsMapper adminToolsMapper, AdminToolsLocationMapper adminToolsLocationMapper,
                                 RedisHashMapAdapter redisHashMapAdapter, SequenceCreater sequenceCreater) {
        this.adminToolsMapper = adminToolsMapper;
        this.adminToolsLocationMapper = adminToolsLocationMapper;
        this.redisHashMapAdapter = redisHashMapAdapter;
        this.sequenceCreater = sequenceCreater;
    }

    @Override
    public PageWarper<ToolsDTO> getToolsList(ToolsParam toolsParam) {

        PageWarper<ToolsDTO> pageWarper = new PageWarper<>(this.adminToolsMapper.queryToolsList(toolsParam));
        pageWarper.getList().forEach(toolsDTO -> {
            if (toolsDTO.getGlobalFlag() == 0) {
                toolsDTO.setLocationList(this.adminToolsLocationMapper.getToolsLocationByToolId(toolsDTO.getId()));
            }
        });
        return pageWarper;
    }

    @Override
    public ToolsDTO getToolsById(Long id) {
        Tools tools = this.adminToolsMapper.selectByPrimaryKey(id);
        return convertTools(tools);
    }

    @Override
    public int updateToolsStatusById(String[] ids, Byte status) {
        removeCache();
        return this.adminToolsMapper.updateToolsStatusById(ids, status);
    }

    @Override
    public int updateToolsStatusById(Long id, Byte status) {
        Tools tools = this.adminToolsMapper.selectByPrimaryKey(id);
        if (tools == null) {
            return 0;
        }
        tools.setStatus(status);
        this.adminToolsMapper.updateByPrimaryKeySelective(tools);
        removeCache();

        return 1;
    }

    @Override
    public int addToolsDetail(ToolsInfoParam toolsInfoParam) {

        //参数转换
        Tools toolInfo = new Tools();
        BeanUtils.copyProperties(toolsInfoParam, toolInfo);

        int operationResult;
        //id为空则为新增
        if (null == toolInfo.getId()) {
            toolInfo.setId(sequenceCreater.nextLongId());
            operationResult = adminToolsMapper.insertSelective(toolInfo);
        } else {
            //否则为更新
            operationResult = adminToolsMapper.updateByPrimaryKeySelective(toolInfo);
        }

        updateToolsLocation(toolInfo.getId(), toolsInfoParam.getToolAreaList());
        removeCache();

        return operationResult;
    }

    @Override
    public int deleteTools(Long id) {
        Tools tools = this.adminToolsMapper.selectByPrimaryKey(id);

        redisHashMapAdapter.remove(this.getToolsListKey());


        return this.adminToolsMapper.deleteByPrimaryKey(tools.getId());
    }

    @Override
    public List<TooltipStrategyDTO> getAllTooltipStrategy() {


        List<TooltipStrategyDTO> resultList = new ArrayList<>();
        //遍历枚举值
        for (TooltipStrategyEnum tooltipStrategyEnum : TooltipStrategyEnum.values()) {

            TooltipStrategyDTO tooltipStrategyDTO = new TooltipStrategyDTO();
            tooltipStrategyDTO.setName(tooltipStrategyEnum.getName());
            tooltipStrategyDTO.setDescription(tooltipStrategyEnum.getDescription());
            resultList.add(tooltipStrategyDTO);
        }


        return resultList;
    }

    /**
     * 更新对应地区的信息
     * @param toolId
     * @param toolAreaList
     */
    private void updateToolsLocation(Long toolId, List<ToolsLocation> toolAreaList) {
        if (CollectionUtils.isEmpty(toolAreaList)) {
            return;
        }

        this.adminToolsLocationMapper.deleteByToolId(toolId);
        insertToolsLocation(toolAreaList, toolId);
    }

    /**
     * 添加地区信息
     * @param toolAreaList 工具地区信息列表
     * @param toolId 工具id
     */
    private void insertToolsLocation(List<ToolsLocation> toolAreaList, Long toolId) {

        //初始化待插入的数据列表
        List<ToolsLocation> toolsLocations = new ArrayList<>(toolAreaList.size());

        toolAreaList.forEach(toolAreaInfo -> {
            ToolsLocation toolsLocation = new ToolsLocation();
            toolsLocation.setId(sequenceCreater.nextLongId());
            toolsLocation.setAreaCode(toolAreaInfo.getAreaCode());
            toolsLocation.setToolId(toolId);
            toolsLocation.setAreaType(toolAreaInfo.getAreaType());

            //塞入待插入数据的列表
            toolsLocations.add(toolsLocation);
        });

        this.adminToolsLocationMapper.insertToolsLocations(toolsLocations);
    }

    /**
     * 获取工具类列表的key
     * @return
     */
    private KeyGenerator getToolsListKey() {
        return RedisConfig.THIRDPARTY_AREACODE_TOOLS.copy().appendKey("list");
    }

    /**
     * 移除对应的redis缓存
     * @param areaCodes
     */
    private void removeRedisInfo(String areaCodes) {
        if (StringUtils.isBlank(areaCodes)) {
            return;
        }
        String[] areaCode = StringUtils.split(areaCodes, ",");
        redisHashMapAdapter.remove(this.getToolsListKey(), areaCode);
    }

    private ToolsDTO convertTools(Tools tools) {
        ToolsDTO toolsDTO = new ToolsDTO();
        BeanUtils.copyProperties(tools, toolsDTO);

        toolsDTO.setLocationList(adminToolsLocationMapper.getToolsLocationByToolId(tools.getId()));


        return toolsDTO;
    }

    /**
     * 移除缓存
     */
    private void removeCache() {
        redisHashMapAdapter.remove(this.getToolsListKey());
    }
}
