package com.bxm.localnews.admin.service.forum.impl;

import com.alibaba.fastjson.JSONObject;
import com.bxm.component.mybatis.utils.MybatisBatchBuilder;
import com.bxm.localnews.admin.constant.RedisConfig;
import com.bxm.localnews.admin.domain.LocationMapper;
import com.bxm.localnews.admin.dto.AreaRelationDTO;
import com.bxm.localnews.admin.param.TopicParam;
import com.bxm.localnews.admin.service.forum.TopicService;
import com.bxm.localnews.admin.service.security.BaseAdminService;
import com.bxm.localnews.admin.vo.Location;
import com.bxm.localnews.admin.vo.Topic;
import com.bxm.localnews.news.domain.AreaRelationMapper;
import com.bxm.localnews.news.domain.MixedRecommendPoolMapper;
import com.bxm.localnews.news.domain.TopicMapper;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.uuid.SequenceCreater;
import com.bxm.newidea.component.vo.PageWarper;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

@Service
public class TopicServiceImpl extends BaseAdminService implements TopicService {

    @Resource
    private TopicMapper topicMapper;

    @Resource
    private AreaRelationMapper areaRelationMapper;

    @Resource
    private LocationMapper locationMapper;

    @Resource
    private SequenceCreater sequenceCreater;

    @Resource
    private RedisStringAdapter redisStringAdapter;

    @Override
    public PageWarper<Topic> getList(TopicParam param) {
        PageWarper<Topic> topicList = new PageWarper<>(topicMapper.getList(param));

        if (CollectionUtils.isNotEmpty(topicList.getList())) {
            for (Topic topic : topicList.getList()) {
                completeTopic(topic);
            }
        }

        return topicList;
    }

    @Override
    public Topic selectByPrimaryKey(Long id) {
        Topic topic = topicMapper.selectByPrimaryKey(id);
        completeTopic(topic);
        return topic;
    }

    @Override
    public int updateStatus(Long id, Byte status) {
        Topic topic = this.topicMapper.selectByPrimaryKey(id);
        if (topic == null) {
            return 0;
        }
        topic.setEnable(status);
        this.topicMapper.updateByPrimaryKeySelective(topic);
        removeCache();

        if ((byte) 2 == status) {
            List<Long> postIdList = this.topicMapper.listPostIdByTopicId(id);
            logger.debug("[删除话题时清除帖子以及混合推荐池],话题id：{},取得话题下所有帖子:{}",id,JSONObject.toJSON(postIdList));
            MybatisBatchBuilder.create(MixedRecommendPoolMapper.class,postIdList).sessionTemplateName("newsSessionTemplate").run(MixedRecommendPoolMapper::deleteByPrimaryKey);
            this.topicMapper.deletePostsByTopicId(id);
        }

        return 1;
    }

    @Override
    public int saveOrUpdate(Topic topic) {
        int update;
        Long id = topic.getId();

        if (null == id) {
            id = sequenceCreater.nextLongId();
            topic.setId(id);
            topic.setCreateTime(new Date());
            update = this.topicMapper.insert(topic);
        } else {
            update = this.topicMapper.updateByPrimaryKeySelective(topic);
        }

        if (CollectionUtils.isNotEmpty(topic.getAreaCodes())) {
            List<AreaRelationDTO> list = new ArrayList<>();
            for (String areaCode : topic.getAreaCodes()) {
                AreaRelationDTO areaRelationDTO = new AreaRelationDTO();
                areaRelationDTO.setId(sequenceCreater.nextLongId());
                areaRelationDTO.setRelationId(topic.getId());
                areaRelationDTO.setAreaCode(areaCode);
                areaRelationDTO.setType((byte) 2);
                list.add(areaRelationDTO);
            }
            areaRelationMapper.batchDelete(topic.getId(), 2);
            areaRelationMapper.batchInsert(list);
        }

        removeCache();

        return update;
    }

    @Override
    public int updateOrder(Long id, Integer order) {
        Topic topic = this.topicMapper.selectByPrimaryKey(id);
        if (topic == null) {
            return 0;
        }
        topic.setOrder(order);
        this.topicMapper.updateByPrimaryKeySelective(topic);
        removeCache();

        return 1;
    }

    @Override
    public List<Topic> getSimpleList() {
        TopicParam param = new TopicParam();
        param.setPageSize(0);
        return topicMapper.getList(param);
    }

    @Override
    public Integer getPostNumByTopicId(Long id) {
        return topicMapper.getPostNumByTopicId(id);
    }

    private void completeTopic(Topic topic) {
        if (topic != null && topic.getIsGlobal() != (byte) 1) {
            topic.setAreaCodes(areaRelationMapper.getAreaCodes(topic.getId(), 2));
            if (CollectionUtils.isNotEmpty(topic.getAreaCodes())) {
                List<Location> areas = new ArrayList<>();
                for (String areaCode : topic.getAreaCodes()) {
                    Location location = locationMapper.selectByCode(areaCode);
                    if (null != location) {
                        areas.add(location);
                    }
                }
                topic.setAreas(areas);
            }
        }
    }

    /**
     * 获取话题列表的key
     *
     * @return
     */
    private KeyGenerator getKey() {
        return RedisConfig.TOPIC.copy();
    }

    /**
     * 移除缓存
     */
    private void removeCache() {
        redisStringAdapter.remove(this.getKey());
    }
}
