package com.bxm.localnews.base.service.impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.bxm.component.mybatis.utils.MybatisBatchBuilder;
import com.bxm.localnews.base.domain.CommonEquipmentAppMapper;
import com.bxm.localnews.base.domain.EquipmentExtraInfoMapper;
import com.bxm.localnews.base.domain.EquipmentInfoMapper;
import com.bxm.localnews.base.service.EquipmentService;
import com.bxm.localnews.common.constant.EquipmentEnum;
import com.bxm.localnews.common.dto.EquipmentDTO;
import com.bxm.localnews.common.param.AppInfoParam;
import com.bxm.localnews.common.param.EquipmentParam;
import com.bxm.localnews.common.vo.CommonEquipmentApp;
import com.bxm.localnews.common.vo.EquipmentExtraInfo;
import com.bxm.localnews.common.vo.EquipmentInfo;
import com.bxm.newidea.component.redis.DistributedLock;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.service.BaseService;
import com.bxm.newidea.component.tools.SpringContextHolder;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Lists;
import org.apache.commons.beanutils.BeanUtils;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.math.NumberUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.retry.RetryException;
import org.springframework.retry.annotation.Backoff;
import org.springframework.retry.annotation.Retryable;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;

import java.lang.reflect.InvocationTargetException;
import java.util.*;
import java.util.stream.Collectors;

import static com.bxm.localnews.common.constant.RedisConfig.EQUIPMENT_CACHE;
import static com.bxm.localnews.common.constant.RedisConfig.POINTS_MALL_KEY;

@Service
public class EquipmentServiceImpl extends BaseService implements EquipmentService {

    private final static String[] BLANK_IDFA = {"0000000000000000", "00000000-0000-0000-0000-000000000000"};

    private final EquipmentInfoMapper equipmentInfoMapper;

    private final EquipmentExtraInfoMapper equipmentExtraInfoMapper;

    private final RedisStringAdapter redisStringAdapter;

    private final DistributedLock distributedLock;

    private final CommonEquipmentAppMapper commonEquipmentAppMapper;

    @Autowired
    public EquipmentServiceImpl(EquipmentInfoMapper equipmentInfoMapper,
                                EquipmentExtraInfoMapper equipmentExtraInfoMapper,
                                RedisStringAdapter redisStringAdapter,
                                DistributedLock distributedLock,
                                CommonEquipmentAppMapper commonEquipmentAppMapper) {
        this.equipmentInfoMapper = equipmentInfoMapper;
        this.equipmentExtraInfoMapper = equipmentExtraInfoMapper;
        this.redisStringAdapter = redisStringAdapter;
        this.distributedLock = distributedLock;
        this.commonEquipmentAppMapper = commonEquipmentAppMapper;
    }

    @Override
    @Retryable(backoff = @Backoff(200), value = RetryException.class)
    public String save(EquipmentParam param) {
        String requestId = nextSequence().toString();
        String key = String.valueOf(param.hashCode());

        if (!distributedLock.lock(key, requestId)) {
            logger.error("存在重复请求，请求参数：{}", JSON.toJSONString(param));
            throw new RetryException("重复请求");
        }

        String devcId = param.getDevcId();

        //如果是数字，说明是新版本的设备ID
        if (NumberUtils.isDigits(devcId)) {
            EquipmentDTO equipment = get(devcId);

            //如果是既有设备，匹配请求参数，判断是否需要更新
            if (null != equipment) {
                execSave(param, equipment);

                distributedLock.unlock(key, requestId);
                return devcId;
            }
        }

        //从数据库进行唯一标识匹配，查找对应的记录，如果能查找到，则返回
        String equipmentId = existsEquipment(param);
        if (equipmentId != null) {
            devcId = equipmentId;
            param.setDevcId(devcId);
            execSave(param, get(devcId));

            distributedLock.unlock(key, requestId);
            return devcId;
        }

        //全新的设备，创建新设备信息
        if (StringUtils.isBlank(devcId) || !NumberUtils.isDigits(devcId)) {
            devcId = String.valueOf(nextId());
            param.setDevcId(devcId);
        }

        execSave(param, null);

        distributedLock.unlock(key, requestId);
        return devcId;
    }

    /**
     * 判断请求的设备是否是新设备
     * @param param 请求参数
     * @return 设备ID
     */
    private String existsEquipment(EquipmentParam param) {
        List<String> identifiers = Lists.newArrayList();
        //兼容历史传错的情况下，把devcid也作为唯一标识
        if (StringUtils.isNotBlank(param.getDevcId())) {
            add(identifiers, param.getDevcId());
        }
        add(identifiers, param.getAndroidId());
        add(identifiers, param.getAndroidUuid());
        add(identifiers, param.getIMEI());
        add(identifiers, param.getIdfa());
        add(identifiers, param.getIosId());

        return equipmentInfoMapper.selectByIdentifiers(identifiers);
    }

    private void add(List<String> identifiers, String unique) {
        if (verify(unique)) {
            identifiers.add(unique);
        }
    }

    private boolean verify(String unique) {
        if (StringUtils.isNotBlank(unique)) {
            return !StringUtils.equalsAny(unique, BLANK_IDFA);
        }
        return false;
    }

    private void execSave(EquipmentParam param, EquipmentDTO exists) {
        //将请求转化为实体对象
        EquipmentDTO request = new EquipmentDTO();
        org.springframework.beans.BeanUtils.copyProperties(param, request);
        request.setId(param.getDevcId());

        //如果两者相同，不做任何处理 (这里复写了equals方法，如果新增了字段需要同步修改)
        if (request.equals(exists)) {
            return;
        }

        //如果对比失败，进行异步信息更新
        SpringContextHolder.getBean(this.getClass()).saveWithAsync(request, exists,param);
    }

    @Override
    @Async
    public void saveWithAsync(EquipmentDTO requestEquipment, EquipmentDTO existsEquipment,EquipmentParam equipmentParam) {

        List<CommonEquipmentApp> commonEquipmentAppList = new ArrayList<>();
        if (com.bxm.newidea.component.tools.StringUtils.isNotEmpty(equipmentParam.getAppInfoParamList())) {
            List<AppInfoParam> appInfoParamList = JSONObject.parseArray(equipmentParam.getAppInfoParamList(), AppInfoParam.class);
            commonEquipmentAppList = appInfoParamList
                    .stream().map(e -> {
                        CommonEquipmentApp commonEquipmentApp = new CommonEquipmentApp();
                        commonEquipmentApp.setAppName(e.getAppName());
                        commonEquipmentApp.setPackageName(e.getPackageName());
                        commonEquipmentApp.setId(Long.valueOf(equipmentParam.getDevcId()));
                        commonEquipmentApp.setModifyTime(new Date());
                        commonEquipmentApp.setCreateTime(new Date());
                        return commonEquipmentApp;
                    }).collect(Collectors.toList());
        }

        //新增设备信息
        if (null == existsEquipment) {
            EquipmentExtraInfo extraInfo = extract(requestEquipment);
            if (StringUtils.isBlank(requestEquipment.getId())) {
                extraInfo.setId(String.valueOf(nextId()));
            }
            extraInfo.setCreateTime(new Date());
            equipmentExtraInfoMapper.insert(extraInfo);
            //增加安装列表信息
            addAppInfo(commonEquipmentAppList);
            requestEquipment.setId(extraInfo.getId());
            saveUniqueInfo(requestEquipment);
        } else {
            //如果是基础信息变更
            if (!requestEquipment.baseInfoEquals(existsEquipment)) {
                EquipmentExtraInfo extraInfo = extract(requestEquipment);
                extraInfo.setId(existsEquipment.getId());
                extraInfo.setModifyTime(new Date());
                equipmentExtraInfoMapper.updateByPrimaryKeySelective(extraInfo);
            }
            //如果唯一标识信息变更
            if (!requestEquipment.uniqueEquals(existsEquipment)) {
                equipmentInfoMapper.deleteById(existsEquipment.getId());
                requestEquipment.setId(existsEquipment.getId());
                saveUniqueInfo(requestEquipment);
            }

            //对比设备安装列表
            if(isNotEquals(existsEquipment.getApps(), commonEquipmentAppList)){
                commonEquipmentAppMapper.delete(Long.valueOf(equipmentParam.getDevcId()));
                MybatisBatchBuilder.create(CommonEquipmentAppMapper.class, commonEquipmentAppList).run(CommonEquipmentAppMapper::insert);
                logger.debug("设备安装信息列表变更：{}", requestEquipment);
            }
        }

        redisStringAdapter.remove(build(requestEquipment.getId()));
    }

    /**
     * 安卓增加设备应用信息
     * @param commonEquipmentAppList
     */
    private void addAppInfo(List<CommonEquipmentApp> commonEquipmentAppList){
        //增加设备应用信息
        MybatisBatchBuilder.create(CommonEquipmentAppMapper.class, commonEquipmentAppList).run(CommonEquipmentAppMapper::insert);
    }

    private boolean isNotEquals(List source, List target) {
        if (null == source && null == target) {
            return false;
        }

        if (null == source || null == target) {
            return true;
        }

        return !CollectionUtils.isEqualCollection(source, target);
    }

    /**
     * 保存设备唯一标识信息
     * @param equipment 设备全部信息
     */
    private void saveUniqueInfo(EquipmentDTO equipment) {
        List<EquipmentInfo> uniqueList = Lists.newArrayList();
        addEquipmentInfo(uniqueList, equipment.getAndroidId(), EquipmentEnum.ANDROID_ID, equipment.getId());
        addEquipmentInfo(uniqueList, equipment.getAndroidUuid(), EquipmentEnum.ANDROID_UUID, equipment.getId());
        addEquipmentInfo(uniqueList, equipment.getIMEI(), EquipmentEnum.ANDROID_IMEI, equipment.getId());
        addEquipmentInfo(uniqueList, equipment.getIdfa(), EquipmentEnum.IOS_IDFA, equipment.getId());
        addEquipmentInfo(uniqueList, equipment.getIosId(), EquipmentEnum.IOS_DEVICE_ID, equipment.getId());

        equipmentInfoMapper.insert(uniqueList);
    }

    /**
     * 组装设备唯一标识信息
     * @param uniqueList    存储唯一标识的集合
     * @param unique        标识值
     * @param equipmentType 标识类型
     */
    private void addEquipmentInfo(List<EquipmentInfo> uniqueList,
                                  String unique,
                                  EquipmentEnum equipmentType,
                                  String equipmentId) {
        if (verify(unique)) {
            EquipmentInfo info = EquipmentInfo.builder()
                    .id(equipmentId)
                    .identifierName(equipmentType.name())
                    .identifierValue(unique)
                    .createTime(new Date())
                    .build();

            uniqueList.add(info);
        }
    }

    /**
     * 从设备实体中提取设备基础信息
     * @param equipment 设备全部信息
     * @return 设备基础信息
     */
    private EquipmentExtraInfo extract(EquipmentDTO equipment) {
        EquipmentExtraInfo extraInfo = new EquipmentExtraInfo();
        org.springframework.beans.BeanUtils.copyProperties(equipment, extraInfo);
        extraInfo.setCurrentVersion(equipment.getCurVer());

        return extraInfo;
    }

    @Override
    public EquipmentDTO getByDeviceId(String deviceId) {
        // 首先作为设备id(服务端生成)查询设备信息
        EquipmentDTO equipmentDTO = get(deviceId);

        // 不为空则直接返回
        if (Objects.nonNull(equipmentDTO)) {
            return equipmentDTO;
        }

        // 否则作为设备id(app的设备信息) 以identify value查询设备id（服务端生成）
        String equipmentId = equipmentInfoMapper.selectByIdentifiers(ImmutableList.of(deviceId));

        // 如果查询到了设备id(服务端生成) 则查询完成的设备信息
        if (null != equipmentId) {
            return get(equipmentId);
        }

        return null;
    }

    private KeyGenerator build(String id) {
        return EQUIPMENT_CACHE.copy().appendKey(id);
    }

    @Override
    public EquipmentDTO get(String id) {
        KeyGenerator key = build(id);
        EquipmentDTO equipment = redisStringAdapter.get(key, EquipmentDTO.class);

        if (equipment == null) {
            EquipmentExtraInfo equipmentExtraInfo = equipmentExtraInfoMapper.selectByPrimaryKey(id);
            if (null == equipmentExtraInfo) {
                return null;
            }

            equipment = new EquipmentDTO();
            equipment.setId(equipmentExtraInfo.getId());
            equipment.setEnableNotification(equipmentExtraInfo.getEnableNotification());
            equipment.setCurVer(equipmentExtraInfo.getCurrentVersion());
            equipment.setOperatingSystem(equipmentExtraInfo.getOperatingSystem());
            equipment.setPhoneModel(equipmentExtraInfo.getPhoneModel());
            equipment.setPlatform(equipmentExtraInfo.getPlatform());

            List<EquipmentInfo> equipmentInfos = equipmentInfoMapper.selectByPrimaryKey(id);
            if (CollectionUtils.isNotEmpty(equipmentInfos)) {
                //转换为map
                Map<String, String> map = new HashMap<>(3);
                equipmentInfos.forEach(equipmentInfo ->
                        map.put(EquipmentEnum.valueOf(equipmentInfo.getIdentifierName()).getType(),
                                equipmentInfo.getIdentifierValue()));
                try {
                    //转换为对象
                    BeanUtils.populate(equipment, map);
                } catch (IllegalAccessException | InvocationTargetException e) {
                    logger.error(e.getMessage(), e);
                }
            }

         //设备安装列表
            equipment.setApps(commonEquipmentAppMapper.getEquipmentApps(Long.valueOf(id)));

            redisStringAdapter.set(key, equipment);
        }
        return equipment;
    }
}
