package com.bxm.localnews.mq.common.model.dto;

import com.alibaba.fastjson.JSONObject;
import com.bxm.localnews.mq.common.constant.MessageBehaviorEnum;
import com.bxm.localnews.mq.common.constant.PushMessageEnum;
import com.bxm.newidea.component.vo.BaseBean;
import com.google.common.collect.Maps;
import io.swagger.annotations.ApiModel;
import io.swagger.annotations.ApiModelProperty;
import org.apache.commons.lang3.math.NumberUtils;

import java.util.Map;

/**
 * 消息负荷对象，用于存放推送、用户事件中对消息的具体定义
 *
 * @author liujia
 */
@ApiModel(description = "消息主体内容")
public class PushPayloadInfo extends BaseBean {

    @ApiModelProperty("当前推送消息的ID")
    private Long msgId;

    @ApiModelProperty("推送的消息类型,详见文档库中的相关定义")
    private int type = 0;

    @ApiModelProperty("创建消息的时间(时间戳)")
    private Long createTime;

    @ApiModelProperty("消息标题")
    private String title;

    @ApiModelProperty("消息内容")
    private String content;

    @ApiModelProperty("通知使用的铃声")
    private String sound;

    /**
     * 跳转协议，历史版本中从extend中获取url参数
     * 新版本使用此字段进行统一的协议地址跳转，不再需要判断type中的值
     * 如果指定了协议，强制设定类型为10，这里是为了兼容历史逻辑
     */
    @ApiModelProperty("跳转协议 以前是把url放在 extend中，往后可以直接放到这个字段，客户端拿到直接跳转")
    private String protocol;

    @ApiModelProperty("是否需要在消息中心显示查看详情，如果为null则不做处理，按照历史逻辑处理，如果为true则显示查看详情，如果为false则不显示")
    private Boolean hasDetail;

    @ApiModelProperty("额外参数，根据当前的推送类型会有不同的返回值")
    private Map<String, Object> extend = Maps.newHashMap();

    public static PushPayloadInfo build() {
        return new PushPayloadInfo().setCreateTime(System.currentTimeMillis());
    }

    public static PushPayloadInfo build(PushMessageEnum type) {
        PushPayloadInfo info = new PushPayloadInfo();
        info.setType(type.getType());
        info.setCreateTime(System.currentTimeMillis());
        return info;
    }

    public Long getTargetUserId() {
        Object userIdObj = getExtend().get("userId");
        if (null == userIdObj) {
            return null;
        }
        if (userIdObj instanceof Long) {
            return Long.valueOf(userIdObj.toString());
        }

        return NumberUtils.toLong(userIdObj.toString());
    }

    public void addTargetUserId(Long userId) {
        addExtend("userId", userId);
    }

    public PushPayloadInfo addExtend(String key, Object value) {
        if (this.extend == null) {
            this.extend = Maps.newHashMap();
        }

        this.extend.put(key, value);
        return this;
    }

    public Long getMsgId() {
        return msgId;
    }

    public void setMsgId(Long msgId) {
        this.msgId = msgId;
    }

    public String toJsonString() {
        return JSONObject.toJSONString(this);
    }

    public Map<String, Object> getExtend() {
        return this.extend;
    }

    public void setExtend(Map<String, Object> extend) {
        this.extend = extend;
    }

    public int getType() {
        return this.type;
    }

    public PushPayloadInfo setType(int type) {
        this.type = type;
        PushMessageEnum messageType = PushMessageEnum.getMessageByType(type);
        if (messageType != null) {
            this.hasDetail = MessageBehaviorEnum.JUMP.equals(messageType.getBehavior());
        }
        return this;
    }

    public Long getCreateTime() {
        return this.createTime;
    }

    public PushPayloadInfo setCreateTime(Long createTime) {
        this.createTime = createTime;
        return this;
    }

    public String getTitle() {
        return this.title;
    }

    public PushPayloadInfo setTitle(String title) {
        this.title = title;
        return this;
    }

    public String getContent() {
        return this.content;
    }

    public PushPayloadInfo setContent(String content) {
        this.content = content;
        return this;
    }

    public String getSound() {
        return this.sound;
    }

    public PushPayloadInfo setSound(String sound) {
        this.sound = sound;
        return this;
    }

    public String getProtocol() {
        return protocol;
    }

    public PushPayloadInfo setProtocol(String protocol) {
        this.protocol = protocol;
        return this;
    }

    public Boolean getHasDetail() {
        return hasDetail;
    }

    public PushPayloadInfo setHasDetail(Boolean hasDetail) {
        this.hasDetail = hasDetail;
        return this;
    }
}
