package com.bxm.localnews.mq.common.model.dto;

import com.bxm.localnews.mq.common.constant.PlatformTypeEnum;
import com.bxm.localnews.mq.common.constant.PushReceiverRuleEnum;
import com.bxm.localnews.mq.common.constant.ReceiverRuleParamConstant;
import com.bxm.newidea.component.vo.BaseBean;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

import java.util.List;
import java.util.Map;

/**
 * 推送消息接收范围
 * 多个接收范围之间是互斥关系,一个消息只能选择一个推送范围
 * 优先级详见：ORDER
 */
public class PushReceiveScope extends BaseBean {
    /**
     * 接收消息的用户
     * ORDER:1
     */
    private Long userId;

    /**
     * 接收消息的用户列表（最多1000条）
     * ORDER:2
     */
    private List<Long> userIds;

    /**
     * 用户接收范围自定义规则类型
     * ORDER:3
     */
    private PushReceiverRuleEnum pushReceiverRule;

    /**
     * 自定义规则对应的参数
     */
    private Map<String, Object> ruleParam;

    /**
     * 限制推送的平台
     */
    private List<PlatformTypeEnum> platformTypes;

    private PushReceiveScope() {
        //私有构造函数，防止构建不符合规范的请求
    }

    /**
     * 推送到个人用户
     *
     * @param userId 用户ID
     * @return 推送接收范围
     */
    public static PushReceiveScope pushSignle(Long userId) {
        return new PushReceiveScope().setPushReceiverRule(PushReceiverRuleEnum.DEFAULT)
                .setUserId(userId)
                .addRuleParam(ReceiverRuleParamConstant.USER_IDS, ImmutableList.of(userId).asList());
    }

    /**
     * 推送到分组用户（指定的用户群）
     *
     * @param userIds 用户列表
     * @return 推送接收范围
     */
    public static PushReceiveScope pushGroup(List<Long> userIds) {
        return new PushReceiveScope().setPushReceiverRule(PushReceiverRuleEnum.DEFAULT)
                .setUserIds(userIds)
                .addRuleParam(ReceiverRuleParamConstant.USER_IDS, userIds);
    }

    /**
     * 根据特殊的规则进行推送
     *
     * @param pushReceiverRule 规则类型
     * @return 推送接收范围
     */
    public static PushReceiveScope pushRule(PushReceiverRuleEnum pushReceiverRule) {
        return new PushReceiveScope().setPushReceiverRule(pushReceiverRule);
    }

    public List<PlatformTypeEnum> getPlatformTypes() {
        return platformTypes;
    }

    public PushReceiveScope setPlatformTypes(List<PlatformTypeEnum> platformTypes) {
        this.platformTypes = platformTypes;
        return this;
    }

    public PushReceiverRuleEnum getPushReceiverRule() {
        return pushReceiverRule;
    }

    public PushReceiveScope setPushReceiverRule(PushReceiverRuleEnum pushReceiverRule) {
        this.pushReceiverRule = pushReceiverRule;
        return this;
    }

    /**
     * 推送用户如果是特殊规则，可以添加参数
     *
     * @param key   特殊规则的key
     * @param value 特殊规则的value
     * @return 推送接收范围
     */
    public PushReceiveScope addRuleParam(String key, Object value) {
        if (null == ruleParam) {
            ruleParam = Maps.newHashMap();
        }
        ruleParam.put(key, value);
        return this;
    }

    public Map<String, Object> getRuleParam() {
        return ruleParam;
    }

    public PushReceiveScope setRuleParam(Map<String, Object> ruleParam) {
        this.ruleParam = ruleParam;
        return this;
    }

    public Long getUserId() {
        return userId;
    }

    public PushReceiveScope setUserId(Long userId) {
        this.userId = userId;
        return this;
    }

    public List<Long> getUserIds() {
        if (userIds == null) {
            userIds = Lists.newArrayList();
        }

        if (userId != null) {
            if (!userIds.contains(userId)) {
                userIds.add(userId);
            }
        }

        return userIds;
    }

    public PushReceiveScope setUserIds(List<Long> userIds) {
        this.userIds = userIds;
        return this;
    }
}
