package com.bxm.localnews.mq.consume.service.impl;

import com.bxm.localnews.mq.common.constant.PushMessageEnum;
import com.bxm.localnews.mq.common.constant.TemplateTypeEnum;
import com.bxm.localnews.mq.common.model.dto.PushMessage;
import com.bxm.localnews.mq.consume.config.AppPushProperties;
import com.bxm.localnews.mq.consume.service.GeTuiService;
import com.gexin.rp.sdk.base.IPushResult;
import com.gexin.rp.sdk.base.ITemplate;
import com.gexin.rp.sdk.base.impl.AppMessage;
import com.gexin.rp.sdk.base.impl.ListMessage;
import com.gexin.rp.sdk.base.impl.SingleMessage;
import com.gexin.rp.sdk.base.impl.Target;
import com.gexin.rp.sdk.base.payload.APNPayload;
import com.gexin.rp.sdk.base.uitls.AppConditions;
import com.gexin.rp.sdk.http.IGtPush;
import com.gexin.rp.sdk.template.NotificationTemplate;
import com.gexin.rp.sdk.template.TransmissionTemplate;
import com.gexin.rp.sdk.template.style.Style0;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.List;

import static com.bxm.localnews.mq.common.constant.PushSoundEnum.IOS_MUTE;

/**
 * Created by Arei on 2018/10/15.
 */
@Service("geiTuiService")
public class GeTuiServiceImpl implements GeTuiService {
    private static final Logger logger = LoggerFactory.getLogger(GeTuiServiceImpl.class);

    @Resource
    private AppPushProperties appPushProperties;

    @Override
    public void push(PushMessage pushMsg) {
        System.setProperty("gexin.rp.sdk.pushlist.needDetails", "true");
        pushMsg.syncPayloadInfo();

        IGtPush push = this.getIgtpush();

        ITemplate template = transmissionTemplate(pushMsg);

        SingleMessage message = new SingleMessage();
        message.setOffline(true);
        // 离线有效时间，单位为毫秒，可选
        message.setOfflineExpireTime(24 * 3600 * 1000);
        message.setData(template);
        // 可选，1为wifi，0为不限制网络环境。根据手机处于的网络情况，决定是否下发
        message.setPushNetWorkType(0);

        Target target = new Target();
        target.setAppId(this.appPushProperties.getId());
        target.setAlias(pushMsg.getAlias());

        IPushResult ret = push.pushMessageToSingle(message, target);
        logger.debug("[{}],push result[{}]", pushMsg, ret.getResponse());

    }

    @Override
    public void pushMsgToAll(PushMessage pushMessage) {


        IGtPush push = this.getIgtpush();
        NotificationTemplate template = this.notificationTemplate(pushMessage);
        AppMessage message = new AppMessage();
        message.setData(template);
        message.setOffline(true);
        message.setOfflineExpireTime(24 * 1000 * 3600);

        List<String> appIdList = new ArrayList<>();
        appIdList.add(this.appPushProperties.getId());

        List<String> phoneTypeList = new ArrayList<>();
        phoneTypeList.add("ANDROID");
        phoneTypeList.add("IOS");

        message.setAppIdList(appIdList);

        AppConditions cdt = new AppConditions();
        cdt.addCondition(AppConditions.PHONE_TYPE, phoneTypeList, AppConditions.OptType.or);
        message.setConditions(cdt);

        IPushResult ret = push.pushMessageToApp(message);

        logger.debug("群发推送----" + ret.getResponse().toString());
    }

    @Override
    public void pushGroupMsg(PushMessage pushMessage) {
        // 配置返回每个用户返回用户状态，可选
        System.setProperty("gexin_pushList_needDetails", "true");
        pushMessage.syncPayloadInfo();
        // 配置返回每个别名及其对应cid的用户状态，可选
        // System.setProperty("gexin_pushList_needAliasDetails", "true");
        IGtPush push = this.getIgtpush();
        // 通知透传模板
        ITemplate template = transmissionTemplate(pushMessage);
        ListMessage message = new ListMessage();
        message.setData(template);
        // 设置消息离线，并设置离线时间
        message.setOffline(true);
        // 离线有效时间，单位为毫秒，可选
        message.setOfflineExpireTime(24 * 1000 * 3600);
        // 配置推送目标
        List targets = getTargets(pushMessage.getUserIds());
        // taskId用于在推送时去查找对应的message
        String taskId = push.getContentId(message);
        IPushResult ret = push.pushMessageToList(taskId, targets);
        logger.debug("批量推送---" + ret.getResponse().toString());
    }

    /**
     * 获取IGtPush实例
     *
     * @return
     */
    private IGtPush getIgtpush() {
        return new IGtPush(this.appPushProperties.getHost(), this.appPushProperties.getKey(), this.appPushProperties.getMasterSecret());
    }

    /**
     * 获取推送目标
     *
     * @param userIds 需要推送的用户列表
     * @return
     */
    private List getTargets(List<String> userIds) {
        List targets = new ArrayList();
        String appId = appPushProperties.getId();

        for (String userId : userIds) {
            Target target = new Target();
            target.setAppId(appId);
            target.setAlias(userId);
            targets.add(target);
        }
        return targets;
    }

    /**
     * 构建IOS消息体
     *
     * @param pushMessage
     * @return
     */
    private APNPayload buildApnPayload(PushMessage pushMessage) {
        APNPayload payload = new APNPayload();
        payload.setAutoBadge("+1");
        if (TemplateTypeEnum.NOTIFCTION.equals(pushMessage.getType())) {
            payload.setContentAvailable(0);
        } else {
            payload.setContentAvailable(1);
        }
        if (pushMessage.isMute()) {
            payload.setSound(IOS_MUTE.getIosSound());
        } else {
            payload.setSound(pushMessage.getSound().getIosSound());
        }

        APNPayload.DictionaryAlertMsg dictionaryAlertMsg = new APNPayload.DictionaryAlertMsg();
        if (!PushMessageEnum.getMessageSilenceByType(pushMessage.getPayloadInfo().getType())) {
            dictionaryAlertMsg.setTitle(pushMessage.getTitle());
            dictionaryAlertMsg.setBody(pushMessage.getContent());
        } else {
            payload.setContentAvailable(0);
        }

        payload.setAlertMsg(dictionaryAlertMsg);
        payload.addCustomMsg("payload", pushMessage.getPayloadInfo().toJsonString());

        return payload;
    }

    /**
     * 构建透传的消息模板
     *
     * @param pushMessage 推送消息参数
     * @return 透传消息模板
     */
    private TransmissionTemplate transmissionTemplate(PushMessage pushMessage) {
        TransmissionTemplate template = new TransmissionTemplate();
        template.setAppId(this.appPushProperties.getId());
        template.setAppkey(this.appPushProperties.getKey());

        // 透传消息设置，1为强制启动应用，客户端接收到消息后就会立即启动应用；2为等待应用启动
        template.setTransmissionType(2);
        template.setTransmissionContent(pushMessage.getPayloadInfo().toJsonString());
        template.setAPNInfo(buildApnPayload(pushMessage));
        return template;
    }

    /**
     * 构建通知消息模板
     *
     * @param pushMessage 推送消息参数
     * @return 通知消息模板
     */
    private NotificationTemplate notificationTemplate(PushMessage pushMessage) {
        NotificationTemplate template = new NotificationTemplate();
        template.setAppId(this.appPushProperties.getId());
        template.setAppkey(this.appPushProperties.getKey());
        template.setTransmissionType(1);
        template.setTransmissionContent(pushMessage.getPayloadInfo().toJsonString());
        template.setAPNInfo(buildApnPayload(pushMessage));

        Style0 style = new Style0();
        style.setTitle(pushMessage.getTitle());
        style.setText(pushMessage.getContent());
        style.setRing(!pushMessage.isMute());
        style.setVibrate(true);
        style.setClearable(true);

        template.setStyle(style);
        return template;
    }
}
