package com.bxm.localnews.msg.listener;

import com.bxm.localnews.mq.common.model.dto.PushMessage;
import com.bxm.localnews.msg.config.PushMessageStatusEnum;
import com.bxm.localnews.msg.param.PushMessageBucket;
import com.bxm.localnews.msg.push.PushExecutor;
import com.bxm.localnews.msg.service.MessageGroupService;
import com.bxm.localnews.msg.stream.MessageProcessor;
import com.bxm.newidea.component.thread.NamedThreadFactory;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.cloud.stream.annotation.StreamListener;
import org.springframework.scheduling.concurrent.ThreadPoolTaskExecutor;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.util.Objects;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;

/**
 * 推送消息的监听器
 *
 * @author liujia
 * @date 2020-07-05 10:25
 **/
@Component
@Slf4j
public class PushMsgListener implements InitializingBean {

    @Resource
    private PushExecutor pushExecutor;

    @Resource
    private MessageGroupService messageGroupService;

    /**
     * 消息推送处理线程池
     */
    private ThreadPoolTaskExecutor executor;

    /**
     * 达到线程池最大线程数量后，统计延后处理的总时间，可以预估延后的总时间
     */
    private AtomicLong totalDelayMills = new AtomicLong(0L);

    /**
     * 消费延迟时间，放缓消费队列的处理速度，使其堆积在消息队列
     */
    private static final long DELAY_MILLS = 100L;

    /**
     * 单条消息的输入处理
     * 消息允许部分丢失，可以采用autoack
     *
     * @param message 从通道获取到的消息
     */
    @StreamListener(MessageProcessor.SINGLE_PUSH_INPUT)
    public void handlerSingle(PushMessage message) {
        // 单条消息的处理逻辑，直接调用发送流程处理
        if (log.isDebugEnabled()) {
            log.info("接收到推送消息：{}", message);
        }

        delay();

        executor.execute(() -> pushExecutor.push(message));
    }

    /**
     * 达到最大处理线程数量，延缓MQ的消费速度
     */
    private void delay() {
        log.debug("excutor thread num:" + executor.getActiveCount());
        if (!hasIdleTime()) {
            try {
                TimeUnit.MILLISECONDS.sleep(DELAY_MILLS);

                long currentDelayMills = totalDelayMills.addAndGet(DELAY_MILLS);
                log.info("触发延缓消息推送速度,当前已延缓总时间：{}", currentDelayMills);
            } catch (InterruptedException e) {
                log.error(e.getMessage(), e);
                Thread.currentThread().interrupt();
            }
        } else {
            totalDelayMills.set(0L);
        }
    }

    /**
     * 当前处理线程是否空闲
     *
     * @return true表示当前服务处理处于空余状态
     */
    public boolean hasIdleTime() {
        return executor.getActiveCount() < executor.getMaxPoolSize();
    }

    /**
     * 批量推送消息的输入处理
     *
     * @param bucket 批量消息对象
     */
    @StreamListener(MessageProcessor.BATCH_PUSH_INPUT)
    public void handlerBatch(PushMessageBucket bucket) {
        // 多条消息的处理逻辑，将其拆分为单独的消息，与单条消息走同一个处理逻辑
        if (log.isDebugEnabled()) {
            log.debug("接收到批量推送：{}", bucket);
        }

        if (null == bucket.getPushMsgId() || CollectionUtils.isEmpty(bucket.getTargetUserIds())) {
            log.error("接受数据不完整，参数为：{}", bucket);
            return;
        }

        // 根据消息ID获取消息
        PushMessage message = messageGroupService.loadCache(bucket.getPushMsgId(), false);
        if (null == message) {
            log.error("接受到的推送消息不存在，接受信息为：{}", bucket);
            return;
        }

        if (Objects.equals(bucket.getIndex(), bucket.getTotal())) {
            if (log.isDebugEnabled()) {
                log.debug("消息[{}]已经处理所有分片，变更为更新完成状态", bucket.getPushMsgId());
            }

            messageGroupService.changeStatus(message.getPayloadInfo().getMsgId(), PushMessageStatusEnum.HAS_BEEN_SEND);
        }

        log.info("处理批量推送消息，消息内容：{} - {},推送人数：{}",
                bucket.getPushMsgId(),
                message.getTitle(),
                bucket.getTargetUserIds().size());

        bucket.getTargetUserIds().forEach(userId -> {
            delay();

            executor.execute(() -> {
                PushMessage cloneMessage = PushMessage.build();
                BeanUtils.copyProperties(message, cloneMessage);

                pushExecutor.push(cloneMessage.assign(userId));
            });
        });
    }

    private void initThreadPool() {
        executor = new ThreadPoolTaskExecutor();
        executor.setCorePoolSize(Runtime.getRuntime().availableProcessors() + 2);
        executor.setQueueCapacity(10000);
        executor.setMaxPoolSize(30);
        executor.setRejectedExecutionHandler(new ThreadPoolExecutor.AbortPolicy());
        executor.setThreadFactory(new NamedThreadFactory("batch-push"));

        executor.initialize();
    }

    @Override
    public void afterPropertiesSet() {
        initThreadPool();
    }
}
