package com.bxm.localnews.msg.service.impl;

import com.bxm.localnews.mq.common.constant.*;
import com.bxm.localnews.mq.common.model.dto.PushMessage;
import com.bxm.localnews.mq.common.model.dto.PushPayloadInfo;
import com.bxm.localnews.mq.common.model.dto.PushReceiveScope;
import com.bxm.localnews.msg.vo.MsgGroupPushBean;
import com.google.common.collect.Lists;
import org.springframework.stereotype.Component;

import java.util.Date;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static org.apache.commons.lang3.StringUtils.isNotEmpty;
import static org.apache.commons.lang3.StringUtils.split;

/**
 * 将运营后台发起的消息转化为统一的推送消息对象，进行适配
 */
@Component
public class PushMessageHandler {
    /**
     * 根据数据库中的推送消息转化为推送对象
     *
     * @param groupPush 群推的原始信息
     * @return 通用的推送消息
     */
    public PushMessage generateMessage(MsgGroupPushBean groupPush) {
        PushMessage pushMessage = PushMessage.build()
                .setTitle(groupPush.getTitle())
                .setContent(groupPush.getContent())
                .setMute(byteToBoolean(groupPush.getIsMute()));

        pushMessage.setType(getTemplateType(groupPush.getPushType()));
        pushMessage.setFloatNotify(byteToBoolean(groupPush.getIsFloatNotify()));
        pushMessage.setPushFoldType(getPushFold(groupPush.getUnfoldType()));
        pushMessage.setPeriodTime(groupPush.getPeriodTime());
        pushMessage.setShock(byteToBoolean(groupPush.getIsVibrate()));
        pushMessage.setActionType(groupPush.getActionType());

        if (isNotEmpty(groupPush.getNotifyImgUrl())) {
            pushMessage.setNotifyImageUrl(groupPush.getNotifyImgUrl());
        }

        if (isNotEmpty(groupPush.getNotifyText())) {
            pushMessage.setNotifyText(groupPush.getNotifyText());
        }

        pushMessage.setPushReceiveScope(getPushReceiveScope(groupPush));
        pushMessage.setStatistics(true);

        pushMessage.setPayloadInfo(getPushInfo(groupPush));

        return pushMessage;
    }

    /**
     * byte值转换为boolean值
     *
     * @param value
     * @return
     */
    private Boolean byteToBoolean(Byte value) {
        return null != value && value != 0;
    }

    /**
     * 通知方式
     */
    private PushFoldTypeEnum getPushFold(String unfoldType) {
        try {
            return PushFoldTypeEnum.valueOf(unfoldType);
        } catch (NullPointerException e) {
            // except catch
        }
        return PushFoldTypeEnum.TEXT;
    }

    /**
     * 推送方式
     */
    private TemplateTypeEnum getTemplateType(String pushType) {
        try {
            return TemplateTypeEnum.valueOf(pushType);
        } catch (NullPointerException e) {
            // except catch
        }
        return TemplateTypeEnum.NOTIFCTION;
    }

    /**
     * 组装参数
     */
    private PushPayloadInfo getPushInfo(MsgGroupPushBean adminPushMessage) {
        Date now = new Date();

        PushPayloadInfo payloadInfo = PushPayloadInfo.build().setTitle(adminPushMessage.getTitle())
                .setContent(adminPushMessage.getContent())
                .setCreateTime(now.getTime());

        payloadInfo.setMsgId(adminPushMessage.getId());
        payloadInfo.addExtend("msgId", adminPushMessage.getId());

        payloadInfo.addExtend("title", adminPushMessage.getTitle());
        // 新闻和帖子仍然兼容旧版本
        if (ActionTypeEnum.OPEN_NEWS.getType().equals(adminPushMessage.getActionType())) {
            payloadInfo.setType(PushMessageEnum.NEWS.getType());
            payloadInfo.addExtend("newsId", adminPushMessage.getActionParams());
            payloadInfo.addExtend("type", 1);
        }

        if (ActionTypeEnum.OPEN_POST.getType().equals(adminPushMessage.getActionType())) {
            payloadInfo.setType(PushMessageEnum.POST.getType());
            payloadInfo.addExtend("postId", Long.valueOf(adminPushMessage.getActionParams()));
        }

        // 使用协议地址，涵盖了后续所有的推送处理，不再扩展类型
        if (ActionTypeEnum.OPEN_URL.getType().equals(adminPushMessage.getActionType())) {
            payloadInfo.setType(PushMessageEnum.OPEN_URL.getType());

            payloadInfo.setProtocol(adminPushMessage.getActionParams());
            payloadInfo.addExtend("url", adminPushMessage.getActionParams());
        }

        return payloadInfo;
    }

    /**
     * 设定推送的接受对象
     *
     * @return 接受消息的范围圈定参数
     */
    private PushReceiveScope getPushReceiveScope(MsgGroupPushBean adminPushMessage) {
        // 限制推送平台
        List<PlatformTypeEnum> platformTypeEnumList = Lists.newArrayList();

        if (isNotEmpty(adminPushMessage.getPushPlatform())) {
            String[] platforms = adminPushMessage.getPushPlatform().split(",");
            for (String platform : platforms) {
                platformTypeEnumList.add(PlatformTypeEnum.valueOf(platform));
            }
        }

        // 推送特定的用户
        if (adminPushMessage.getPushTarget() == 1) {
            String[] userIds = split(adminPushMessage.getTargetUserid(), ",");
            return PushReceiveScope.pushGroup(Stream.of(userIds).map(Long::new).collect(Collectors.toList()))
                    .setPlatformTypes(platformTypeEnumList);
        }

        // 如果是带有条件的推送，通过设定的条件圈定用户
        if (adminPushMessage.getPushTarget() == 2 || adminPushMessage.getPushTarget() == 3) {
            PushReceiveScope pushReceiveScope = PushReceiveScope.pushRule(PushReceiverRuleEnum.CONDITION)
                    .setPlatformTypes(platformTypeEnumList);

            String[] areas = split(adminPushMessage.getAreaScope(), ",");
            if (null != areas && areas.length > 0) {
                pushReceiveScope.addRuleParam(ReceiverRuleParamConstant.AREA_CODES, Stream.of(areas).collect(Collectors.toList()));
            }

            // 额外的扩展参数，解决运营后台、用户服务之间参数传递的问题，具体协定根据业务调整而定义
            pushReceiveScope.addRuleParam("extend", adminPushMessage.getExtendParams());

            return pushReceiveScope;
        }

        return null;
    }
}
