package com.bxm.localnews.msg.service.impl;

import com.bxm.localnews.mq.common.model.dto.UserTokenParam;
import com.bxm.localnews.msg.domain.MsgUserTokenMapper;
import com.bxm.localnews.msg.param.UserLogoutEvent;
import com.bxm.localnews.msg.service.UserTokenService;
import com.bxm.localnews.msg.stream.MessageProcessor;
import com.bxm.localnews.msg.vo.MsgUserTokenBean;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.uuid.SequenceCreater;
import com.fasterxml.jackson.core.type.TypeReference;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cloud.stream.annotation.StreamListener;
import org.springframework.stereotype.Service;

import java.util.Date;
import java.util.List;

import static com.bxm.localnews.msg.constant.RedisConfig.USER_TOKEN_CACHE;

@Service
@Slf4j
public class UserTokenServiceImpl implements UserTokenService {

    private final MsgUserTokenMapper msgUserTokenMapper;

    private final RedisStringAdapter redisStringAdapter;

    private final SequenceCreater sequenceCreater;

    private TypeReference<List<MsgUserTokenBean>> reference = new TypeReference<List<MsgUserTokenBean>>() {
    };

    /**
     * 用户token缓存过期时间
     */
    private static final long EXPIRED_SECONDS = 3 * 3600 * 24L;

    @StreamListener(MessageProcessor.USER_LOGOUT_INPUT)
    public void handlerLogout(UserLogoutEvent event) {
        if (null == event) {
            return;
        }

        if (log.isDebugEnabled()) {
            log.debug("清理用户token，用户ID：{}", event.getUserId());
        }

        msgUserTokenMapper.deleteByUserId(event.getUserId());
        removeCache(event.getUserId());
    }

    @Autowired
    public UserTokenServiceImpl(MsgUserTokenMapper msgUserTokenMapper,
                                RedisStringAdapter redisStringAdapter,
                                SequenceCreater sequenceCreater) {
        this.msgUserTokenMapper = msgUserTokenMapper;
        this.redisStringAdapter = redisStringAdapter;
        this.sequenceCreater = sequenceCreater;
    }

    private KeyGenerator buildKey(Long userId) {
        return USER_TOKEN_CACHE.copy().appendKey(userId);
    }

    @Override
    public List<MsgUserTokenBean> loadToken(Long userId) {
        KeyGenerator key = buildKey(userId);
        List<MsgUserTokenBean> entries = redisStringAdapter.get(key, reference);

        if (null == entries) {
            List<MsgUserTokenBean> userTokenList = msgUserTokenMapper.queryUserToken(userId);

            if (CollectionUtils.isEmpty(userTokenList)) {
                entries = Lists.newArrayList();
                entries.add(new MsgUserTokenBean());
            } else {
                entries = userTokenList;
            }

            redisStringAdapter.set(key, entries, EXPIRED_SECONDS);
        }
        return entries;
    }

    private void removeCache(Long userId) {
        redisStringAdapter.remove(buildKey(userId));
    }

    @Override
    public void removeToken(MsgUserTokenBean tokenInfo) {
        if (null != tokenInfo.getUserId()) {
            msgUserTokenMapper.deleteByUserId(tokenInfo.getUserId());
            removeCache(tokenInfo.getUserId());
        }
    }

    @Override
    public void addToken(UserTokenParam param) {
        Preconditions.checkArgument(null != param && null != param.getUserId());
        Preconditions.checkArgument(null != param.getTokenType());

        msgUserTokenMapper.removeToken(param.getUserId(), param.getTokenType().getCode());

        MsgUserTokenBean token = new MsgUserTokenBean();
        token.setId(sequenceCreater.nextLongId());
        token.setCreateTime(new Date());
        token.setUserId(param.getUserId());
        token.setPlatform((byte) param.getPlatform().getCode());
        token.setToken(param.getToken());
        token.setType(param.getTokenType().getCode());

        msgUserTokenMapper.insert(token);

        removeCache(param.getUserId());
    }
}




















