package com.bxm.localnews.payment.controller;

import com.bxm.localnews.common.util.ResultUtil;
import com.bxm.localnews.common.vo.Json;
import com.bxm.localnews.payment.constant.OrderTypeEnum;
import com.bxm.localnews.payment.constant.PayTypeEnum;
import com.bxm.localnews.payment.dto.OrderStatusDTO;
import com.bxm.localnews.payment.dto.PaymentOrderDTO;
import com.bxm.localnews.payment.order.PaymentOrderFactory;
import com.bxm.localnews.payment.param.UserOrderParam;
import com.bxm.localnews.payment.pay.PaymentOrderService;
import com.bxm.newidea.component.util.WebUtils;
import com.bxm.newidea.component.vo.ResponseJson;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletRequest;
import javax.validation.Valid;
import java.math.BigDecimal;

@Api(tags = "6-01 支付相关接口", description = "包括支付的相关接口操作")
@RestController
@RequestMapping("/api/pay")
public class PayController {

    private PaymentOrderService payService;

    private final PaymentOrderFactory paymentOrderFactory;

    @Autowired
    public PayController(PaymentOrderService payService,
                         PaymentOrderFactory paymentOrderFactory) {
        this.payService = payService;
        this.paymentOrderFactory = paymentOrderFactory;
    }

    @ApiOperation(value = "6-01-1 [已废弃]统一下单（购买本地VIP），并组装所需支付参数",
            notes = "旧版本购买的本地VIP 请调用6-01-5创建订单")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "userId", value = "用户ID", dataType = "Long", required = true),
            @ApiImplicitParam(name = "payType", value = "付款类型, 1:微信，2：支付宝", allowableValues = "1,2", required = true),
            @ApiImplicitParam(name = "returnUrl", value = "支付成功后跳转地址", required = true)
    })
    @PostMapping("/order")
    @Deprecated
    public Json<PaymentOrderDTO> payOrder(Long userId, Byte payType, String returnUrl, HttpServletRequest request) {
        UserOrderParam param = new UserOrderParam();
        param.setUserId(userId);
        param.setPayType(payType);
        param.setReturnUrl(returnUrl);
        param.setRequestIp(WebUtils.getIpAddr(request));
        param.setOrderType(OrderTypeEnum.LOCAL_VIP.getCode());

        PaymentOrderDTO order = paymentOrderFactory.build(param);
        if (order == null) {
            return ResultUtil.genFailedResult("你已成功激活VIP,请勿重复购买");
        }
        return ResultUtil.genSuccessResult(order);
    }

    @ApiOperation(value = "6-01-2 查询订单信息", notes = "支付后查询订单状态")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "paymentNum",
                    value = "订单编号, 1待付款，2取消付款，3付款成功，4付款失败，5失败超时",
                    dataType = "String",
                    required = true)
    })
    @GetMapping("/order/status")
    public Json<OrderStatusDTO> queryOrder(String paymentNum) {
        return ResultUtil.genSuccessResult(payService.queryOrder(paymentNum));
    }

    @ApiOperation(value = "6-01-3 [已废弃]统一下单，并组装所需支付参数", notes = "请调用6-01-5创建订单")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "userId", value = "用户ID", dataType = "Long", required = true),
            @ApiImplicitParam(name = "payType", value = "付款类型, 1:微信，2：支付宝", allowableValues = "1,2", required = true),
            @ApiImplicitParam(name = "amount", value = "支付金额(金额:元)", required = true),
            @ApiImplicitParam(name = "bizId", value = "业务id"),
    })
    @PostMapping("/order/money")
    @Deprecated
    public Json<PaymentOrderDTO> payOrder(Long userId, Byte payType, @RequestParam("amount") BigDecimal amount,
                                          @RequestParam(value = "bizId", required = false) Long bizId,
                                          HttpServletRequest request) {
        UserOrderParam param = new UserOrderParam();
        param.setUserId(userId);
        param.setPayType(payType);
        param.setAmount(amount);
        param.setBizId(bizId);
        param.setRequestIp(WebUtils.getIpAddr(request));
        param.setOrderType(OrderTypeEnum.RED_PACKET.getCode());

        PaymentOrderDTO order = paymentOrderFactory.build(param);
        if (order == null) {
            return ResultUtil.genFailedResult("下单失败,请稍后尝试");
        }
        return ResultUtil.genSuccessResult(order);
    }

    @ApiOperation(value = "6-01-4 微信APP支付后，查询支付结果", notes = "查询支付结果(只适用于查询微信支付结果)")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "paymentNum",
                    value = "订单编号, 返回的状态值:1待付款，2取消付款，3付款成功，4付款失败，5失败超时",
                    dataType = "String",
                    required = true)
    })
    @GetMapping("/order/wxStatus")
    public Json<OrderStatusDTO> queryWxOrderStatus(String paymentNum) {
        return ResultUtil.genSuccessResult(payService.queryOrderStatus(paymentNum, PayTypeEnum.WX_PAY.getType()));
    }

    @ApiOperation(value = "6-01-5 创建预付订单", notes = "触发支付，创建用户订单。前端通过返回的订单编号轮询订单当前状态进行后续处理")
    @PostMapping("/order/create")
    public ResponseJson<PaymentOrderDTO> createPayOrder(@Valid @RequestBody UserOrderParam param, HttpServletRequest request) {
        param.setRequestIp(WebUtils.getIpAddr(request));
        return ResponseJson.ok(paymentOrderFactory.build(param));
    }
}
