package com.bxm.localnews.thirdparty.controller;

import com.alibaba.fastjson.JSON;
import com.bxm.localnews.common.constant.RedisConfig;
import com.bxm.localnews.common.constant.RespCode;
import com.bxm.localnews.common.vo.Json;
import com.bxm.localnews.thirdparty.config.ThirdPartyProperties;
import com.bxm.localnews.thirdparty.service.MachineService;
import com.bxm.localnews.thirdparty.vo.MachineBaseResponse;
import com.bxm.localnews.thirdparty.vo.MachineResponse;
import com.bxm.localnews.thirdparty.vo.MachineState;
import com.bxm.newidea.component.controller.BaseController;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.tools.ShortUrlGenerator;
import com.bxm.newidea.component.tools.StringUtils;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

/**
 * 纸巾机机器相关调用接口
 */
@Api(tags = "3-04 纸巾机设备相关操作接口", description = "接入第三方纸巾机接口相关操作")
@RestController
@RequestMapping("api/public/machine")
public class MachineController extends BaseController {

    private final ThirdPartyProperties thirdPartyProperties;

    private MachineService machineService;

    private RedisStringAdapter redisStringAdapter;

    @Autowired
    public MachineController(ThirdPartyProperties thirdPartyProperties, MachineService machineService,
                             RedisStringAdapter redisStringAdapter) {
        this.thirdPartyProperties = thirdPartyProperties;
        this.machineService = machineService;
        this.redisStringAdapter = redisStringAdapter;
    }

    @ApiOperation(value = "主平台信息回调", hidden = true)
    @PostMapping("callback")
    public String postCallback(@RequestParam String Nonce, @RequestParam String ParamData,
                               @RequestParam String Signature, @RequestParam String TimeStamp) {
        logger.info("callback nonce:{},paramdata:{},signature:{},timestamp:{}", Nonce,
                ParamData, Signature, TimeStamp);
        MachineBaseResponse response = machineService.sendPaperSuccessCallback(ParamData, TimeStamp, Nonce, Signature);
        return JSON.toJSONString(response);
    }

    @ApiOperation(value = "3-04-1 发放纸巾", notes = "返回[0]表示请求成功，返回[1]表示请求路径错误，返回[2]表示已经领取过，返回[3]表示设备异常或不可访问")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "url", value = "设备上二维码对应的URL"),
            @ApiImplicitParam(name = "devcId", value = "当前请求设备ID")
    })
    @PostMapping("open")
    public Json<MachineState> openPaper(@RequestParam String url, @RequestParam String devcId) {
        //从短链接获取设备号码
        String key = ShortUrlGenerator.getShortKey(url);
        String machineId = redisStringAdapter.get(RedisConfig.MACHINE_SHORT_URL_KEY.copy().setKey(key), String.class);
        if (StringUtils.isBlank(machineId)) {
            logger.warn("接收到错误的请求二维码，地址是：[{}]", url);
            return Json.build(RespCode.OK, "请求地址不正确，不是一个正确的二维码").setResult(new MachineState(1));
        }
        logger.info("current machine:[{}],request url:[{}]", machineId, url);
        return machineService.sendPaper(machineId, devcId);
    }

    @ApiOperation(value = "获取设备状态", hidden = true)
    @GetMapping("status")
    public MachineResponse getMachineState(@RequestParam String machineId) {
        return machineService.getMachineOnlineStatus(machineId);
    }

}
