package com.bxm.localnews.user.controller;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.bxm.localnews.common.util.ResultUtil;
import com.bxm.localnews.common.vo.BasicParam;
import com.bxm.localnews.common.vo.Json;
import com.bxm.localnews.user.constant.RedisConfig;
import com.bxm.localnews.user.enums.AppConst;
import com.bxm.localnews.user.login.LoginService;
import com.bxm.localnews.user.login.adapter.ParamAdapter;
import com.bxm.localnews.user.param.RegisterParam;
import com.bxm.localnews.user.vo.LoginInfo;
import com.bxm.localnews.user.vo.LoginMeta;
import com.bxm.localnews.user.vo.LoginParam;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisSetAdapter;
import com.bxm.newidea.component.util.WebUtils;
import com.bxm.newidea.component.vo.Message;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;

import lombok.extern.slf4j.Slf4j;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RestController;

import java.util.Objects;

import static com.alibaba.fastjson.JSON.toJSONString;

@Slf4j
@Api(tags = "9-01 用户登录相关API", description = "用户登录和注册")
@RestController
@RequestMapping("api/login")
public class LoginController {

    private static final Logger logger = LoggerFactory.getLogger(LoginController.class);

    @Resource
    private LoginService loginService;

    @Resource
    private RedisSetAdapter redisSetAdapter;

    @ApiOperation(value = "9-01-1 用户登录", notes = "支持[手机号密码登录][第三方登录][h5微信授权登录]")
    @ApiImplicitParams({
        @ApiImplicitParam(name = "platform", value = "平台信息 1：安卓 2：IOS 3：H5", allowableValues = "1,2,3"),
        @ApiImplicitParam(name = "chnl", value = "渠道信息 具体查看C_CHANNEL表或找[赵亚东]")
    })
    @RequestMapping(value = "toLogin", method = {RequestMethod.GET, RequestMethod.POST})
    public Json<LoginMeta> login(LoginParam loginParam, BasicParam basicParam, HttpServletRequest request) {
        logger.info("登录参数:{}，基础参数:{}", toJSONString(loginParam), toJSONString(basicParam));

        LoginInfo loginInfo = ParamAdapter.transformLoginParamToLoginInfo(loginParam, basicParam);
        String ip = WebUtils.getIpAddr(request);
        loginInfo.setRegIp(ip);
        loginInfo.setLastLoginIp(ip);

        try {
            Message message = this.loginService.doLogin(loginInfo, basicParam);
            if (message.isSuccess()) {
                LoginMeta loginMeta = message.getParam("meta");
                KeyGenerator k = RedisConfig.IS_NEED_RE_LOGIN.copy();
                if ("3.1.2".equals(basicParam.getCurVer())) {
                    logger.info("3.1.2的用户需要进行记录");
                    if (!redisSetAdapter.exists(k, loginMeta.getUser().getId())) {
                        redisSetAdapter.add(k, loginMeta.getUser().getId());
                    }
                }

                if (Objects.nonNull(message.getParam("meta"))) {
                    if (log.isDebugEnabled()) {
                        log.debug("登录成功，登录参数:{}，基础参数:{}，返回数据: {}", toJSONString(loginParam),
                                toJSONString(basicParam), toJSONString(message));
                    }
                    return ResultUtil.genSuccessResult(message.getParam("meta"));
                }
            }

            // 否则登录失败
            log.error("登录失败，登录参数:{}，基础参数:{}，错误信息: {}", toJSONString(loginParam), toJSONString(basicParam),
                    toJSONString(message));

        } catch(Exception e) {
            log.error("登录错误，登录参数:{}，基础参数:{}", toJSONString(loginParam), toJSONString(basicParam), e);
        }

        return ResultUtil.genFailedResult("登录失败，请稍后再试");
    }

    @ApiOperation(value = "9-01-2 用户注册", notes = "支持[端内手机号注册][端内第三方绑定注册][h5站外微信第三方授权注册]")
    @ApiImplicitParams({
        @ApiImplicitParam(name = "platform", value = "平台信息 1：安卓 2：IOS 3：H5", allowableValues = "1,2,3"),
        @ApiImplicitParam(name = "chnl", value = "渠道信息 具体查看C_CHANNEL表或找[赵亚东]")
    })
    @RequestMapping(value = "registerBindingPhone", method = RequestMethod.POST)
    public Json<LoginMeta> registerAndBindingPhone(RegisterParam registerParam, BasicParam basicParam, Long userId,
        HttpServletRequest request) {
        logger.warn("注册参数:{}", toJSONString(registerParam));
        if (AppConst.LOGIN_TYPE.PHONE == registerParam.getType()) {
            logger.warn("该用户使用手机注册,请注意,用户信息:{}", JSON.toJSON(registerParam));
        }
        LoginInfo loginInfo = ParamAdapter.transformRegisterToLoginInfo(registerParam, basicParam);
        String ip = WebUtils.getIpAddr(request);
        loginInfo.setRegIp(ip);
        loginInfo.setLastLoginIp(ip);
        return this.loginService.registerAndBindingPhone(loginInfo, userId, basicParam);
    }

}
