package com.bxm.localnews.user.controller;

import com.bxm.localnews.base.integration.DomainIntegrationService;
import com.bxm.localnews.common.constant.DomainScene;
import com.bxm.localnews.common.constant.RespCode;
import com.bxm.localnews.common.rediskey.UserRedisKey;
import com.bxm.localnews.common.util.ResultUtil;
import com.bxm.localnews.common.vo.BasicParam;
import com.bxm.localnews.common.vo.Json;
import com.bxm.localnews.user.attribute.UserAttributeService;
import com.bxm.localnews.user.dto.*;
import com.bxm.localnews.user.dto.officialAccount.OfficialAccountsUserInfoDTO;
import com.bxm.localnews.user.event.LogoutActionEvent;
import com.bxm.localnews.user.login.UserRegistrationService;
import com.bxm.localnews.user.login.UserService;
import com.bxm.localnews.user.login.adapter.ParamAdapter;
import com.bxm.localnews.user.param.PushParam;
import com.bxm.localnews.user.param.SilentRegisterParam;
import com.bxm.localnews.user.param.TempUserParam;
import com.bxm.localnews.user.param.UserBindPhoneParam;
import com.bxm.localnews.user.properties.CommonProperties;
import com.bxm.localnews.user.support.UserInternalService;
import com.bxm.localnews.user.vo.LoginInfo;
import com.bxm.localnews.user.vo.LoginMeta;
import com.bxm.localnews.user.vo.UserBindPhoneVO;
import com.bxm.newidea.component.controller.BaseController;
import com.bxm.newidea.component.jwt.tools.JwtTokenUtil;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.tools.StringUtils;
import com.bxm.newidea.component.util.WebUtils;
import com.bxm.newidea.component.vo.Message;
import com.bxm.newidea.component.vo.ResponseJson;
import com.google.common.eventbus.EventBus;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import org.springframework.cloud.context.config.annotation.RefreshScope;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import java.util.Objects;

import static com.bxm.localnews.common.constant.RespCode.BAD_REQUEST;
import static org.apache.commons.lang3.StringUtils.join;

@Api(tags = "9-02 用户相关API", description = "用户注册和管理")
@RestController
@RequestMapping("api/user")
@RefreshScope
public class UserController extends BaseController {

    @Resource
    private UserService userService;

    @Resource
    private UserRegistrationService userRegistrationService;

    @Resource
    private UserInternalService userInternalService;

    @Resource
    private CommonProperties commonProperties;

    @Resource
    private UserAttributeService userAttributeService;

    @Resource
    private RedisHashMapAdapter redisHashMapAdapter;

    @Resource
    private DomainIntegrationService domainIntegrationService;

    @Resource
    private EventBus userActionEventBus;

    @ApiOperation(value = "9-02-1 静默注册", notes = "用户打开app，不存在userId,根据设备号来返回该手机游客模式的userId")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "platform", value = "平台信息 1：安卓 2：IOS 3：H5", allowableValues = "1,2,3"),
            @ApiImplicitParam(name = "chnl", value = "渠道信息 具体查看C_CHANNEL表或找[赵亚东]"),
            @ApiImplicitParam(name = "firstOpenType", value = "是否是第一次打开应用，1：首次打开"),
    })
    @RequestMapping(value = "toRegister", method = RequestMethod.POST)
    public Json<LoginMeta> register(SilentRegisterParam registerParam, BasicParam basicParam, String firstOpenType,
                                    HttpServletRequest request) {
        LoginInfo loginInfo = ParamAdapter.transformSilentRegisterToLoginInfo(registerParam);
        loginInfo.setRegIp(WebUtils.getIpAddr(request));
        //这个方法使用的是LoginInfo的类，但是现在要用别的类了，所以需要使用适配器模式
        return this.userRegistrationService.register(loginInfo, basicParam, firstOpenType);
    }

    @ApiOperation(value = "9-02-3 重置用户token")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "userId", value = "用户id"),
            @ApiImplicitParam(name = "refreshToken", value = "刷新的token")
    })
    @RequestMapping(value = "refreshToken", method = RequestMethod.POST)
    public Json<UserInfoDTO> refreshToken(@RequestParam(value = "userId") long userId,
                                          @RequestParam(value = "refreshToken") String refreshToken) {
        return this.userService.refreshToken(userId, refreshToken);
    }

    @ApiOperation(value = "9-02-5 退出登录")
    @ApiImplicitParam(name = "userId", value = "用户ID")
    @GetMapping(value = "logout")
    public Json logout(Long userId) {
        // 清空用户redis信息
        userService.removeUserCache(userId);
        // 移除用户token
        redisHashMapAdapter.remove(UserRedisKey.HASH_USER_TOKEN, String.valueOf(userId));

        userActionEventBus.post(LogoutActionEvent.of().setUserId(userId));
        return ResultUtil.genSuccessResult();
    }

    @ApiOperation(value = "9-02-12 获取用户支付token", notes = "获取用户的支付token，有效期为一个小时")
    @ApiImplicitParam(name = "userId", value = "用户ID", required = true)
    @PostMapping(value = "getPayToken")
    public Json<PayToken> getPayToken(Long userId) {
        String token = JwtTokenUtil.generateToken(userId);
        return ResultUtil.genSuccessResult(new PayToken(token));
    }

    @ApiOperation(value = "9-02-13 获取用户推广链接（包含用户邀请码）")
    @ApiImplicitParam(name = "userId", value = "用户ID", required = true)
    @RequestMapping(value = "getRegisterUrl", method = RequestMethod.POST)
    public Json<RegisterMeta> getRegisterUrl(Long userId) {
        // 使用落地页为内容类型的分享域名 拼接
        return ResultUtil.genSuccessResult(new RegisterMeta(join(domainIntegrationService.getOutSideShareBaseUrl(
                DomainScene.DomainViewScene.CONTENT_VIEW),
                commonProperties.getChannelPath(),
                userId)));
    }

    @ApiOperation(value = "9-02-14 用户切换城市并更新位置", notes = "根据客户端传过来的区域编号更新用户的地理位置，并添加用户地理位置历史，在每次启动app后调用")
    @PostMapping(value = "/location")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "userId", value = "用户ID"),
            @ApiImplicitParam(name = "code", value = "地区编码")
    })
    public Json<LocationDTO> getUserLocation(@RequestParam(value = "userId", required = false) Long userId,
                                             @RequestParam(value = "code", required = false) String code,
                                             BasicParam basicParam,
                                             HttpServletRequest request) {
        String ip = WebUtils.getIpAddr(request);
        return ResultUtil.genSuccessResult(userInternalService.doPostingLocation(code, userId, basicParam, ip));
    }

    @ApiOperation(value = "9-02-16 获取手机号码是否已注册")
    @GetMapping(value = "/userPhone")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "phone", value = "手机号")
    })
    public Json<Boolean> isUserPhone(@RequestParam(value = "phone") String phone,
                                     @RequestParam(value = "flag", required = false, defaultValue = "0") Integer flag) {
        return Json.build(userService.checkUserExistByPhone(phone, flag));
    }

    @ApiOperation(value = "9-02-17 保存用户推送信息", notes = "每个用户会保存推送的平台以及推送的token")
    @PostMapping(value = "/push")
    public Json getPushToken(PushParam pushParam) {
        if (StringUtils.isEmpty(pushParam.getPushPlatform()) ||
                StringUtils.isEmpty(pushParam.getPushToken()) ||
                null == pushParam.getUserId()) {
            return ResultUtil.genFailedResult(BAD_REQUEST, "参数错误");
        }
        userAttributeService.savePush(pushParam);
        return ResultUtil.genSuccessResult();
    }

    @ApiOperation(value = "9-02-18 获取用户头像的文件地址", notes = "为解决客户端请求微信图片跨域问题")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "userId", value = "用户ID", dataType = "long")
    })
    @GetMapping("/userInfo/headimg")
    public Json<String> getUserHeaderImg(@RequestParam(name = "userId") Long userId) {
        String fileInputStream = userInternalService.getUserHeaderImg(userId);
        if (null == fileInputStream) {
            return Json.build(RespCode.BAD_REQUEST, "获取用户头像文件流错误");
        } else {
            return Json.build(fileInputStream);
        }
    }

    @ApiOperation(value = "9-02-19 获取用户基本信息", notes = "包含 头像、昵称、地址")
    @ApiImplicitParam(name = "userId", value = "用户ID")
    @RequestMapping(value = "getLocationUserInfo", method = RequestMethod.GET)
    public Json<LocationUserInfoDTO> getLocationUserInfo(@RequestParam(value = "userId", required = false) Long userId) {
        if (Objects.isNull(userId)) {
            userId = 0L;
        }
        return ResultUtil.genSuccessResult(this.userService.getLocationUserInfo(userId));
    }

    @ApiOperation(value = "9-02-20 创建临时用户", notes = "APP分享新闻、活动等页面通过邀请途径生成临时用户")
    @PostMapping(value = "/create/temp")
    public Json<WxUserInfo> createTempUser(TempUserParam tempUserParam, HttpServletRequest request) {
        tempUserParam.setRegIp(WebUtils.getIpAddr(request));
        Message message = userService.getWxUserInfo(tempUserParam, request);
        if (message.isSuccess()) {
            return ResultUtil.genSuccessResult(message.getParam("wxUserInfo"));
        } else {
            return ResultUtil.genFailedResult(message.getLastMessage());
        }
    }

    @ApiOperation(value = "9-02-22 获取用户头像、昵称")
    @ApiImplicitParam(name = "currentUserId", value = "用户ID")
    @RequestMapping(value = "getNickNameAndHeadImg", method = RequestMethod.GET)
    public Json<BaseUserInfo> getUserBaseInfo(@RequestParam(value = "currentUserId") Long currentUserId) {
        return ResultUtil.genSuccessResult(this.userService.getNickNameAndHeadImg(currentUserId));
    }

    @ApiOperation(value = "9-02-23 【我的】获取个人相关信息", notes = "包含姓名、头像、金币、签到信息、邀请人数")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "userId", value = "用户ID", required = true),
            @ApiImplicitParam(name = "platform", value = "平台信息 1：安卓 2：IOS 3：H5", allowableValues = "1,2,3", required = true),
    })
    @GetMapping(value = "getUserInfo")
    public Json<UserBaseInfoDTO> getUserInfo(@RequestParam(value = "userId") long userId,
                                             @RequestParam(value = "platform") int platform) {
        return ResultUtil.genSuccessResult(userInternalService.getUserPersonalCenterInfo(userId, platform));
    }

    @ApiOperation(value = "9-02-24 站外校验是否绑定手机号", notes = "只有userId 就校验是否已经绑定，只有phone，就校验是否已经使用,true:可以使用 ，fasle不可以")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "userId", value = "用户ID"),
            @ApiImplicitParam(name = "phone", value = "手机号码")
    })
    @GetMapping(value = "/public/checkHasBindPhone")
    public Json<UserBindPhoneVO> publicCheckHasBindPhone(@RequestParam String phone, @RequestParam Long userId) {
        return ResultUtil.genSuccessResult(userService.checkUserAndPhoneExists(UserBindPhoneParam.builder().phone(phone).userId(userId).build()));
    }


    @ApiOperation(value = "9-02-25 校验是否绑定手机号", notes = "站内用来校验是否绑定手机号码")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "userId", value = "用户ID"),
            @ApiImplicitParam(name = "phone", value = "手机号码")
    })
    @GetMapping(value = "/checkHasBindPhone")
    public Json<UserBindPhoneVO> checkHasBindPhone(@RequestParam String phone, @RequestParam Long userId) {
        return ResultUtil.genSuccessResult(userService.checkUserAndPhoneExists(UserBindPhoneParam.builder().phone(phone).userId(userId).build()));
    }


    @ApiOperation(value = "9-02-26 站外绑定手机号", notes = "站外绑定手机号")
    @PostMapping(value = "/public/bindUserPhone")
    public Json<UserBindPhoneVO> publicBindUserPhone(@RequestBody UserBindPhoneParam userBindPhoneParam) {
        return ResultUtil.genSuccessResult(userService.bindUserPhone(userBindPhoneParam));
    }

    @ApiOperation(value = "9-02-27 站内绑定手机号", notes = "绑定手机号")
    @PostMapping(value = "/bindUserPhone")
    public Json<UserBindPhoneVO> bindUserPhone(@RequestBody UserBindPhoneParam userBindPhoneParam) {
        return ResultUtil.genSuccessResult(userService.bindUserPhone(userBindPhoneParam));
    }


    @ApiOperation(value = "9-02-28 站外公众号获取用户基本信息", notes = "包含 头像、昵称、地址")
    @ApiImplicitParam(name = "userId", value = "用户ID")
    @GetMapping(value = "officialAccountsUserInfo")
    public ResponseJson<OfficialAccountsUserInfoDTO> officialAccountsUserInfo(@RequestParam(value = "userId", required = false) Long userId) {
        if (Objects.isNull(userId)) {
            userId = 0L;
        }
        return ResponseJson.ok(userService.officialAccountsUserInfo(userId));
    }
}
