package com.bxm.localnews.user.param;

import com.bxm.localnews.user.enums.AccountActionEnum;
import com.bxm.localnews.user.enums.CashFlowTypeEnum;
import com.bxm.localnews.user.enums.GoldFlowTypeEnum;
import org.apache.commons.lang3.builder.ToStringBuilder;
import org.apache.commons.lang3.builder.ToStringStyle;

import java.math.BigDecimal;

import static com.bxm.localnews.user.enums.GoldFlowTypeEnum.OTHER;

/**
 * 账号处理参数
 *
 * @author liujia
 * @date 2020/05/04 10:56
 */
public class AccountActionParam {

    /**
     * 防止使用的人构建错误的对象，这里使用私有的构造函数
     */
    private AccountActionParam() {
    }

    /**
     * 处理动作
     */
    private AccountActionEnum action;

    /**
     * 账号所属用户ID
     */
    private Long userId;

    /**
     * 操作金币数（针对金币）
     */
    private Integer goldNum;

    /**
     * 操作金额(针对现金)
     */
    private BigDecimal amount;

    /**
     * 流水备注(eg:)
     */
    private String remark;

    /**
     * 流水详细内容（eg:给[xx帖子]点赞消耗一朵小红花）
     */
    private String content;

    /**
     * 金币流水原因备注（运营操作数据进行记录）
     */
    private String reason;

    /**
     * 本次账号操作关联的业务ID
     */
    private Long relationId;

    /**
     * 3.10.0版本 记录收益相关信息
     *
     * @see com.bxm.localnews.user.model.param.earnings.EarningsSaveOrUpdateParam
     */
    private Object extData;

    /**
     * 金币流水类型（主要是兼容之前的逻辑）
     */
    private GoldFlowTypeEnum goldFlowType;

    /**
     * 现金流水类型（主要是兼容之前的逻辑）
     */
    private CashFlowTypeEnum cashFlowType;

    /**
     * 对用户的金币进行操作
     *
     * @param action 操作动作，具体的业务逻辑由action对应的handler进行处理
     * @return 金币操作构建者
     */
    public static GoldBuilder goldBuilder(AccountActionEnum action) {
        return new GoldBuilder(action);
    }

    /**
     * 构建一个空的金币构建者
     *
     * @return 金币操作构建者
     */
    public static GoldBuilder goldBuilder() {
        return new GoldBuilder();
    }

    /**
     * 将用户的金币转化为现金
     *
     * @param action 操作动作，具体的业务逻辑由action对应的handler进行处理
     * @return 金币转化为现金的操作构建者
     */
    public static GoldToCashBuilder goldToCashBuilder(AccountActionEnum action) {
        return new GoldToCashBuilder(action);
    }

    /**
     * 构建一个空的金币转现金构建者
     *
     * @return 金币转现金操作构建者
     */
    public static GoldToCashBuilder goldToCashBuilder() {
        return new GoldToCashBuilder();
    }

    /**
     * 对用户的现金进行操作
     *
     * @param action 操作动作，具体的业务逻辑由action对应的handler进行处理
     * @return 现金操作构建者
     */
    public static CashBuilder cashBuilder(AccountActionEnum action) {
        return new CashBuilder(action);
    }

    /**
     * 构建一个空的现金构建者
     *
     * @return 现金操作构建者
     */
    public static CashBuilder cashBuilder() {
        return new CashBuilder();
    }

    /**
     * 基础信息接口，规范扩展的构建者必须覆盖的基础属性
     */
    public interface BaseBuilder {

        /**
         * 设置用户
         *
         * @param userId 操作的目标账号
         */
        BaseBuilder userId(Long userId);

        /**
         * 设置操作涉及的关联业务ID
         */
        BaseBuilder relationId(Long relationId);

        /**
         * 操作备注信息，写入流水
         */
        BaseBuilder remark(String remark);

        /**
         * 操作动作信息，用于匹配Handler
         */
        BaseBuilder action(AccountActionEnum action);

        /**
         * 收益扩展字段
         */
        BaseBuilder extData(Object extData);

        /**
         * 构建参数对象
         *
         * @return 参数对象
         */
        AccountActionParam build();
    }

    public static class GoldBuilder implements BaseBuilder {

        AccountActionParam param = new AccountActionParam();

        private GoldBuilder(AccountActionEnum action) {
            param.setAction(action);
        }

        private GoldBuilder() {
        }

        @Override
        public GoldBuilder action(AccountActionEnum action) {
            param.setAction(action);
            return this;
        }

        public GoldBuilder goldNum(Integer goldNum) {
            param.setGoldNum(goldNum);
            return this;
        }

        @Override
        public GoldBuilder userId(Long userId) {
            this.param.setUserId(userId);
            return this;
        }

        @Override
        public GoldBuilder relationId(Long relationId) {
            this.param.setRelationId(relationId);
            return this;
        }

        @Override
        public GoldBuilder remark(String remark) {
            this.param.setRemark(remark);
            return this;
        }

        public GoldBuilder reason(String reason) {
            this.param.setReason(reason);
            return this;
        }

        public GoldBuilder content(String content) {
            this.param.setContent(content);
            return this;
        }

        public GoldBuilder goldFlowType(GoldFlowTypeEnum goldFlowType) {
            this.param.setGoldFlowType(goldFlowType);
            return this;
        }

        @Override
        public BaseBuilder extData(Object extData) {
            return null;
        }

        @Override
        public AccountActionParam build() {
            if (null == this.param.getUserId()
                    || null == this.param.getAction()
                    || null == this.param.getGoldNum()) {
                throw new IllegalArgumentException("action,userId、goldNum不能为空，参数：" + this.param);
            }

            if (this.param.getGoldFlowType() == null) {
                this.param.setGoldFlowType(OTHER);
            }

            return param;
        }
    }

    public static class CashBuilder implements BaseBuilder {

        AccountActionParam param = new AccountActionParam();

        @Override
        public CashBuilder action(AccountActionEnum action) {
            param.setAction(action);
            return this;
        }

        @Override
        public BaseBuilder extData(Object extData) {
            this.param.setExtData(extData);
            return this;
        }

        private CashBuilder(AccountActionEnum action) {
            param.setAction(action);
        }

        private CashBuilder() {
        }

        public CashBuilder amount(BigDecimal amount) {
            param.setAmount(amount);
            return this;
        }

        @Override
        public CashBuilder userId(Long userId) {
            this.param.setUserId(userId);
            return this;
        }

        @Override
        public CashBuilder relationId(Long relationId) {
            this.param.setRelationId(relationId);
            return this;
        }

        @Override
        public CashBuilder remark(String remark) {
            this.param.setRemark(remark);
            return this;
        }

        public CashBuilder cashFlowType(CashFlowTypeEnum cashFlowType) {
            this.param.setCashFlowType(cashFlowType);
            return this;
        }

        @Override
        public AccountActionParam build() {
            if (null == this.param.getUserId()
                    || null == this.param.getAction()
                    || null == this.param.getAmount()) {
                throw new IllegalArgumentException("action,userId、amount不能为空");
            }
            return param;
        }
    }

    public static class GoldToCashBuilder implements BaseBuilder {

        AccountActionParam param = new AccountActionParam();

        private GoldToCashBuilder(AccountActionEnum action) {
            param.setAction(action);
        }

        private GoldToCashBuilder() {
        }

        @Override
        public GoldToCashBuilder action(AccountActionEnum action) {
            param.setAction(action);
            return this;
        }

        @Override
        public BaseBuilder extData(Object extData) {
            return this;
        }

        public GoldToCashBuilder goldNum(Integer goldNum) {
            param.setGoldNum(goldNum);
            return this;
        }

        public GoldToCashBuilder amount(BigDecimal amout) {
            param.setAmount(amout);
            return this;
        }

        @Override
        public GoldToCashBuilder userId(Long userId) {
            this.param.setUserId(userId);
            return this;
        }

        @Override
        public GoldToCashBuilder relationId(Long relationId) {
            this.param.setRelationId(relationId);
            return this;
        }

        @Override
        public GoldToCashBuilder remark(String remark) {
            this.param.setRemark(remark);
            return this;
        }

        @Override
        public AccountActionParam build() {
            if (null == this.param.getUserId()
                    || null == this.param.getAction()
                    || null == this.param.getGoldNum()
                    || null == this.param.getAmount()) {
                throw new IllegalArgumentException("userId、targetUserId、amount不能为空,参数：" + this.param);
            }
            return param;
        }
    }

    public AccountActionEnum getAction() {
        return action;
    }

    private void setAction(AccountActionEnum action) {
        this.action = action;
    }

    public Long getUserId() {
        return userId;
    }

    private void setUserId(Long userId) {
        this.userId = userId;
    }

    public Integer getGoldNum() {
        return goldNum;
    }

    private void setGoldNum(Integer goldNum) {
        this.goldNum = goldNum;
    }

    public BigDecimal getAmount() {
        return amount;
    }

    private void setAmount(BigDecimal amount) {
        this.amount = amount;
    }

    public String getRemark() {
        return remark;
    }

    private void setRemark(String remark) {
        this.remark = remark;
    }

    public Long getRelationId() {
        return relationId;
    }

    private void setRelationId(Long relationId) {
        this.relationId = relationId;
    }

    public String getContent() {
        return content;
    }

    private void setContent(String content) {
        this.content = content;
    }

    public GoldFlowTypeEnum getGoldFlowType() {
        return goldFlowType;
    }

    private void setGoldFlowType(GoldFlowTypeEnum goldFlowType) {
        this.goldFlowType = goldFlowType;
    }

    public CashFlowTypeEnum getCashFlowType() {
        return cashFlowType;
    }

    private void setCashFlowType(CashFlowTypeEnum cashFlowType) {
        this.cashFlowType = cashFlowType;
    }

    public String getReason() {
        return reason;
    }

    private void setReason(String reason) {
        this.reason = reason;
    }

    public Object getExtData() {
        return extData;
    }

    public void setExtData(Object extData) {
        this.extData = extData;
    }

    @Override
    public String toString() {
        return ToStringBuilder.reflectionToString(this, ToStringStyle.JSON_STYLE);
    }
}
