package com.bxm.localnews.user.service.barrels.weight;


import com.bxm.localnews.user.constant.RedisConfig;
import com.bxm.localnews.user.dto.UserInfoDTO;
import com.bxm.localnews.user.enums.RecommendCategoryEnum;
import com.bxm.localnews.user.param.NativeRecommendContext;
import com.bxm.localnews.user.param.RecommendNativeParam;
import com.bxm.localnews.user.param.WeightParam;
import com.bxm.localnews.user.properties.RecommendProperties;
import com.bxm.localnews.user.service.BlockUserService;
import com.bxm.localnews.user.service.barrels.BarrelSupport;
import com.bxm.localnews.user.service.handler.WeightAlternateBarrelHandler;
import com.bxm.localnews.user.service.handler.WeightDefaultBarrelHandler;
import com.bxm.localnews.user.service.handler.WeightPreferredBarrelHandler;
import com.bxm.localnews.user.vo.RecommendNative;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisSetAdapter;
import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.Multimap;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.time.LocalDate;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.stream.Collectors;

import static com.alibaba.fastjson.JSON.toJSON;

@BarrelSupport(value = {
        WeightPreferredBarrelHandler.class,
        WeightDefaultBarrelHandler.class,
        WeightAlternateBarrelHandler.class})
@Service
public class WeightDataPreloadNativeBarrel extends AbstractWeightBarrelChoose {

    @Autowired
    private BlockUserService blockUserService;

    @Autowired
    private RedisSetAdapter redisSetAdapter;

    @Autowired
    private RecommendProperties recommendProperties;

    /**
     * 所有前置的判断
     * 生成查询实体
     *
     * @param nativeRecommendContext
     * @return
     */
    @Override
    public Boolean doInvoke(NativeRecommendContext nativeRecommendContext) {
        //其他推荐是没有下拉的
        if (RecommendCategoryEnum.OTHER.getName().equals(nativeRecommendContext.getRecommendCategory()) && 1 == nativeRecommendContext.getActionType()) {
            return true;
        }

        //如果是下滑的话填充查询参数
        if (1 == nativeRecommendContext.getActionType()) {
            UserInfoDTO user = nativeRecommendContext.getParam(USER_INFO);

            //适配新查询实体
            RecommendNativeParam recommendNativeParam = RecommendNativeParam.builder()
                    .ownSex(user.getSex() == null ? 0 : user.getSex())
                    .lat(nativeRecommendContext.getLat())
                    .lon(nativeRecommendContext.getLon())
                    .currentAreaCode(nativeRecommendContext.getCurrentAreaCode())
                    .hometownCode(user.getHometownCode() == null ? user.getLocationCode() : user.getHometownCode())
                    .locationCode(user.getLocationCode())
                    .recommendCategory(nativeRecommendContext.getRecommendCategory())
                    .userId(nativeRecommendContext.getUserId()).build();
            logger.debug("本地人推荐->通用参数:[{}]", toJSON(recommendNativeParam));

            //如果是筛选，则拼接筛选参数
            if (RecommendCategoryEnum.FILTER.getName().equals(nativeRecommendContext.getRecommendCategory())) {
                generateFilterParam(nativeRecommendContext, recommendNativeParam);
                logger.debug("本地人推荐->筛选参数:[{}]", toJSON(recommendNativeParam));
            }

            //将查询参数访问上下文中
            nativeRecommendContext.addParam(RECOMMEND_NATIVE_PARAM, recommendNativeParam);

            //填充参数
            WeightParam weightParam = nativeRecommendContext.getParam("weight");
            recommendNativeParam.setWeightParam(weightParam);
            logger.debug("本地人推荐->权重参数:[{}]", toJSON(weightParam));

            //这里有一步执行多个数据准备，相互无关系
            CompletableFuture<Void> future1
                    = CompletableFuture.runAsync(() -> {
                //准备数据过滤原始数据（降权与拉黑）
                Map<Long, Integer> blockUserMap = blockUserService.listBlockUser();
                logger.debug("本地人推荐->获取拉黑或者降权的用户map:[{}]", toJSON(blockUserMap));
                nativeRecommendContext.addParam(USER_BLOCK, blockUserMap);
            });
            CompletableFuture<Void> future2
                    = CompletableFuture.runAsync(() -> {
                //准备数据过滤原始数据（忽略）
                Map<Long, Integer> ignoreUserMap = recommendProperties.getIgnoreUserList().parallelStream()
                        .collect(Collectors.toMap(Long::longValue, e -> 1, (key1, key2) -> key1));
                logger.debug("本地人推荐->获取忽略的用户map:[{}]", toJSON(ignoreUserMap));
                nativeRecommendContext.addParam(USER_IGNORE, ignoreUserMap);
            });
            CompletableFuture<Void> future3
                    = CompletableFuture.runAsync(() -> {
                //准备已经曝光过的数据（去曝光）
                Multimap<Long, RecommendNative> recommendNativeMap = getRecommendCache(nativeRecommendContext);
                logger.debug("本地人推荐->获取已经曝光过的用户map:[{}]", toJSON(recommendNativeMap));
                nativeRecommendContext.addParam(USER_CACHE, recommendNativeMap);
            });

            CompletableFuture<Void> all = CompletableFuture.allOf(future1, future2, future3);
            try {
                all.get();
                logger.info("【拉黑或者降权】、【忽略】、【曝光】准备数据就绪");
            } catch (InterruptedException | ExecutionException e) {
                logger.error("【拉黑或者降权】、【忽略】、【曝光】准备数据获取出错[{}]", e.getMessage());
                logger.error(e.getMessage(), e);
                Thread.currentThread().interrupt();
            }

        }
        return false;
    }

    /**
     * 填补筛选参数
     *
     * @param nativeRecommendContext
     * @param recommendNativeParam
     */
    private void generateFilterParam(NativeRecommendContext nativeRecommendContext, RecommendNativeParam recommendNativeParam) {
        //当年龄是18-60的时候不筛选，我也很无奈啊
        boolean needQuery = null != nativeRecommendContext.getMaxAge() && null != nativeRecommendContext.getMinAge()
                && !(60 == nativeRecommendContext.getMaxAge() && 18 == nativeRecommendContext.getMinAge());

        if (needQuery) {
            //得到最大年龄的出生日期
            recommendNativeParam.setStartBirth(LocalDate.now().minusYears(nativeRecommendContext.getMaxAge()));

            //得到最小年龄的出生日期
            recommendNativeParam.setEndBirth(LocalDate.now().minusYears(nativeRecommendContext.getMinAge()));
        }
        //得到筛选的性别
        if (null != nativeRecommendContext.getSex()) {
            recommendNativeParam.setSex(nativeRecommendContext.getSex());
        }
        //得到筛选的行业
        if (null != nativeRecommendContext.getIndustryId()) {
            recommendNativeParam.setJobCategory(nativeRecommendContext.getIndustryId());
        }
    }


    /**
     * 获得本地人推荐的缓存数据
     *
     * @param nativeRecommendContext
     * @return
     */
    private Multimap<Long, RecommendNative> getRecommendCache(NativeRecommendContext nativeRecommendContext) {
        Multimap<Long, RecommendNative> multimap = ArrayListMultimap.create();
        KeyGenerator recommendKey = RedisConfig.USER_RECOMMEND.copy()
                .appendKey("recommended")
                .appendKey(nativeRecommendContext.getRecommendCategory())
                .appendKey(nativeRecommendContext.getUserId());
        if (Boolean.FALSE.equals(redisSetAdapter.hasKey(recommendKey))) {
            return multimap;
        }
        Set<RecommendNative> recommendNativeList = redisSetAdapter.getAllMembers(recommendKey, RecommendNative.class);
        recommendNativeList.forEach(e -> multimap.put(e.getUserId(), e));
        return multimap;
    }
}
