package com.bxm.spider.cache;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;

import java.util.HashSet;
import java.util.Set;

/**
 * <p> redis常用方法工具类 </p>
 *
 * @Author: JandMin
 * @since: 1.0.0
 * @Date: 2018/10/20
 */
@Component
public class RedisClient {
    private static final Logger LOG = LoggerFactory.getLogger(RedisClient.class);

    @Autowired
    @Qualifier("jedisPool")
    private  JedisPool jedisPool;

  /*  public RedisClient(){
        RedisConfiguration redisConfig = new RedisConfiguration();
        this.jedisPool = new JedisPool(redisConfig,redisConfig.getHost(),redisConfig.getPort(),redisConfig.getTimeout(),
                redisConfig.getPassword(),redisConfig.getDatabase());
    }

    public static void main(String[] args) {
        RedisClient redisClient = new RedisClient();

        String key = "a:b:c:";
        System.out.println(" begin -->" + redisClient.dimCount(key,null));
        Long result = redisClient.dimDel(key,null);
        System.out.println(result);
        System.out.println(" end -->" + redisClient.dimCount(key,null));

       System.exit(0);
    }*/
    
    /**
     * 模糊删除redis key,根据 key + * 匹配
     * @date: 2019/2/19
     * @param key
     * @return java.lang.Long
     * @throws 
     */
    public Long dimDel(String key){
        return dimDel(key,null);
    }
    /**
     * 模糊删除redis key,根据 key + * 匹配
     * @date: 2019/2/19
     * @param key
     * @param dbIndex
     * @return java.lang.Long
     * @throws 
     */
    public Long dimDel(String key, Integer dbIndex){
        Long result = 0L;
        Jedis jedis = null;
        try {
            jedis = jedisPool.getResource();
            if (null != dbIndex && dbIndex.intValue() >= 0) {
                jedis.select(dbIndex);
            }
            Set<String> keys = jedis.keys(key+"*");
            for(String k : keys){
                System.out.println(k);
            }
            String[] delKeys = new String[keys.size()];
            result = jedis.del(keys.toArray(delKeys));
        } catch (Exception e){
            if(LOG.isErrorEnabled()){
                LOG.error("hGet the key error,the key is {}",key, e);
            }
        } finally {
            if(null != jedis){
                jedis.close();
            }
        }
        return result;
    }
    /**
     * 根据 key 模糊统计 set 的数量
     * @date: 2019/2/19
     * @param key
     * @return java.lang.Integer
     * @throws 
     */
    public Long dimCount(String key){
        return dimCount(key,null);
    }
    /**
     * 根据 key 模糊统计 set 的数量
     * @date: 2019/2/19
     * @param key
     * @param dbIndex
     * @return java.lang.Long
     * @throws 
     */
    public Long dimCount(String key, Integer dbIndex){
        Long size = 0L;
        Jedis jedis = null;
        try {
            jedis = jedisPool.getResource();
            if (null != dbIndex && dbIndex.intValue() >= 0) {
                jedis.select(dbIndex);
            }
            Set<String> keys = jedis.keys(key + "*");
            for(String k : keys){
                size += scard(k);
            }
        } catch (Exception e){
            if(LOG.isErrorEnabled()){
                LOG.error("dimCount the key error,the key is {}",key, e);
            }
        } finally {
            if(null != jedis){
                jedis.close();
            }
        }
        return size;
    }
    /**
     * 获取哈希表中指定字段的值
     * @date: 2018/10/20
     * @param key
     * @param field
     * @return java.lang.String
     * @throws 
     */
    public String hGet(String key,String field){
        return hGet(key, field,null);
    }
    /**
     * 获取哈希表中指定字段的值
     * @date: 2018/10/20
     * @param key
     * @param field
     * @param dbIndex
     * @return java.lang.String
     * @throws 
     */
    public String hGet(String key,String field,Integer dbIndex){
        if(StringUtils.isBlank(key) || StringUtils.isBlank(field)){
            if(LOG.isWarnEnabled()){
                LOG.warn("the key or field is null");
            }
            return null;
        }
        Jedis jedis = null;
        try {
            jedis = jedisPool.getResource();
            if (null != dbIndex && dbIndex.intValue() >= 0) {
                jedis.select(dbIndex);
            }
            return jedis.hget(key,field);
        } catch (Exception e){
            if(LOG.isErrorEnabled()){
                LOG.error("hGet the key error,the key is {},the field is {}",key,field, e);
            }
        } finally {
            if(null != jedis){
                jedis.close();
            }
        }
        return null;
    }

    /**
     * hash 删除
     * @date: 2018/10/20
     * @param key
     * @param field
     * @return java.lang.String
     * @throws
     */
    public void hdel(String key,String field){
        hdel(key, field,null);
    }
    /**
     * hash 删除-指定库
     * @date: 2018/10/20
     * @param key
     * @param field
     * @param dbIndex
     * @return java.lang.String
     * @throws
     */
    public void hdel(String key, String field, Integer dbIndex){
        if(StringUtils.isBlank(key) || StringUtils.isBlank(field)){
            if(LOG.isWarnEnabled()){
                LOG.warn("the key or field is null");
            }
        }
        Jedis jedis = null;
        try {
            jedis = jedisPool.getResource();
            if (null != dbIndex && dbIndex.intValue() >= 0) {
                jedis.select(dbIndex);
            }
            jedis.hdel(key, field);
        } catch (Exception e){
            if(LOG.isErrorEnabled()){
                LOG.error("hGet the key error,the key is {},the field is {}",key,field, e);
            }
        } finally {
            if(null != jedis){
                jedis.close();
            }
        }
    }
    /**
     * 判断redis中是否存在key
     *
     * @param key
     * @return boolean
     * @throws
     * @author kk.xie
     * @date 2018/11/27 15:39
     */
    public boolean exists(String key){
        return exists(key, null);
    }

    /**
     * 判断redis中是否存在key
     *
     * @param key
     * @param dbIndex
     * @return boolean
     * @throws
     * @author kk.xie
     * @date 2018/11/27 15:39
     */
    public boolean exists(String key, Integer dbIndex){
        if(StringUtils.isBlank(key)){
            if(LOG.isWarnEnabled()){
                LOG.warn("the key is null");
            }
        }
        Jedis jedis = null;
        try {
            jedis = jedisPool.getResource();
            if (null != dbIndex && dbIndex.intValue() >= 0) {
                jedis.select(dbIndex);
            }
            return jedis.exists(key);
        } catch (Exception e){
            if(LOG.isErrorEnabled()){
                LOG.error("exists the key error,the key is {}",key , e);
            }
        } finally {
            if(null != jedis){
                jedis.close();
            }
        }
        return false;
    }

    /**
     * hash 先取后删
     *
     * @param key
     * @param field
     * @return java.lang.String
     * @throws
     * @author kk.xie
     * @date 2018/11/20 11:03
     */
    public String hgetAnddel(String key,String field){
        return hgetAnddel(key, field, null);
    }

    /**
     * hash 先取后删-指定库
     *
     * @param key
     * @param field
     * @param dbIndex
     * @return java.lang.String
     * @throws
     * @author kk.xie
     * @date 2018/11/20 11:03
     */
    public String hgetAnddel(String key, String field, Integer dbIndex){
        String result = hGet(key, field, dbIndex);
        hdel(key, field, dbIndex);
        return result;
    }

    /**
     * hash 字段自增 1
     *
     * @param key
     * @param field
     * @return java.lang.Long
     * @throws
     * @author kk.xie
     * @date 2018/10/25 15:37
     */
    public Long hincrByOne(String key,String field){
        return hincrByOne(key, field, null);
    }

    /**
     * hash 字段自增 1
     *
     * @param key
     * @param field
     * @param dbIndex
     * @return java.lang.Long
     * @throws
     * @author kk.xie
     * @date 2018/10/25 15:36
     */
    public Long hincrByOne(String key,String field,Integer dbIndex){
        if(StringUtils.isBlank(key) || StringUtils.isBlank(field)){
            if(LOG.isWarnEnabled()){
                LOG.warn("the key or field is null");
            }
            return null;
        }
        Jedis jedis = null;
        try {
            jedis = jedisPool.getResource();
            if (null != dbIndex && dbIndex.intValue() >= 0) {
                jedis.select(dbIndex);
            }
            return jedis.hincrBy(key,field, 1);
        } catch (Exception e){
            if(LOG.isErrorEnabled()){
                LOG.error("hincrByOne the key error,the key is {},the field is {}",key,field, e);
            }
        } finally {
            if(null != jedis){
                jedis.close();
            }
        }
        return null;
    }

    /**
     * 为哈希表中的字段赋值
     * @date: 2018/10/20
     * @param key
     * @param field
     * @param value
     * @return void
     * @throws 
     */
    public void hSet(String key,String field,String value){
        hSet( key, field, value, null,null);
    }
    /**
     * 为哈希表中的字段赋值
     * @date: 2018/10/20
     * @param key
     * @param field
     * @param value
     * @param dbIndex
     * @return void
     * @throws 
     */
    public void hSet(String key,String field,String value,Integer dbIndex){
        hSet( key, field, value, dbIndex,null);
    }
    /**
     * 为哈希表中的字段赋值
     * @date: 2018/10/20
     * @param key
     * @param field
     * @param value
     * @param dbIndex
     * @param expire
     * @return void
     * @throws 
     */
    public  void hSet(String key,String field,String value,Integer dbIndex,Integer expire){
        if(StringUtils.isBlank(key) || StringUtils.isBlank(field)){
            if(LOG.isWarnEnabled()){
                LOG.warn("the key or field is null");
            }
        }
        Jedis jedis = null;
        try {
            jedis = jedisPool.getResource();
            if (null != dbIndex && dbIndex.intValue() >= 0) {
                jedis.select(dbIndex);
            }
            jedis.hset(key, field, value);
            if (null != expire && expire.intValue() >= 0) {
                jedis.expire(key, expire);
            }
        } catch (Exception e){
            if(LOG.isErrorEnabled()){
                LOG.error("hSet the key error,the key is {},the field is {}, the value is {}",key,field,value, e);
            }
        } finally {
            if(null != jedis) {
                jedis.close();
            }
        }
    }
    /**
     * 保存 key-value
     *
     * @param key
     * @param value
     * @return void
     * @throws
     * @author kk.xie
     * @date 2018/10/19 17:31
     */
    public  void set(String key, String value){
        set(key, value, null);
    }

    /**
     * 保存 key-value
     *
     * @param dbIndex 数据库
     * @param key
     * @param value
     * @return void
     * @throws
     * @author kk.xie
     * @date 2018/10/19 17:32
     */
    public void set(String key, String value, Integer dbIndex) {
        Jedis jedis = null;
        try {
            jedis = jedisPool.getResource();
            if (dbIndex != null) {
                jedis.select(dbIndex);
            }
            jedis.set(key, value);
        } catch (Exception e){
            if(LOG.isErrorEnabled()){
                LOG.error("set the key error,db {} the key is {} the value is {}",dbIndex, key ,value, e);
            }
        }finally {
            if (jedis != null) {
                jedis.close();
            }
        }
    }

    /**
     * 取值 key-value
     *
     * @param key
     * @return java.lang.String
     * @throws
     * @author kk.xie
     * @date 2018/10/19 17:33
     */
    public String get(String key){
        return get(key, null);
    }

    /**
     * 取值 key-value
     *
     * @param dbIndex 数据库
     * @param key
     * @return java.lang.String
     * @throws
     * @author kk.xie
     * @date 2018/10/19 17:34
     */
    public String get(String key, Integer dbIndex) {
        Jedis jedis = null;
        try {
            jedis = jedisPool.getResource();
            if (dbIndex != null) {
                jedis.select(dbIndex);
            }
            return jedis.get(key);
        }  catch (Exception e){
            if(LOG.isErrorEnabled()){
                LOG.error("get the key error,db {} the key is {} the value is {}",dbIndex, key, e);
            }
        }finally {
            if (jedis != null) {
                jedis.close();
            }
        }
        return null;
    }

    /**
     * 保存 key-set
     *
     * @param key
     * @param values
     * @return void
     * @throws
     * @author kk.xie
     * @date 2018/10/19 17:51
     */
    public void sadd(String key, String... values){
        sadd( key,null, values);
    }
    public void sadd(String key, Integer dbIndex, String... values){
        sadd( key,null,null, values);
    }

    /**
     * 保存 key-set 指定数据库
     *
     * @param dbIndex 数据库
     * @param key
     * @param values
     * @return void
     * @throws
     * @author kk.xie
     * @date 2018/10/19 17:51
     */
    public void sadd(String key, Integer dbIndex, Integer expire, String... values){
        Jedis jedis = null;
        try {
            jedis = jedisPool.getResource();
            if (dbIndex != null) {
                jedis.select(dbIndex);
            }
            jedis.sadd(key, values);
            if (null != expire && expire.intValue() > 0) {
                jedis.expire(key, expire);
            }
        } catch (Exception e){
            LOG.error("sadd the key error,db {} the key is {}",dbIndex, key, e);
        }finally {
            if (jedis != null) {
                jedis.close();
            }
        }
    }
    /**
     * 同时新增到两个key中，第一个key有过期实际，第二个key无过期时间
     * @date: 2019/1/10
     * @param dayKey 有过期时间的key
     * @param totalKey 无过期时间的key
     * @param value 添加的值
     * @param dbIndex 数据库选择
     * @param expire 第一个key值得过期时间
     * @return void
     * @throws 
     */
    public void sadd(String dayKey,String totalKey, String value, Integer dbIndex, Integer expire){
        Jedis jedis = null;
        try {
            jedis = jedisPool.getResource();
            if (dbIndex != null) {
                jedis.select(dbIndex);
            }
            jedis.sadd(dayKey, value);
            if (null != expire && expire.intValue() > 0) {
                jedis.expire(dayKey, expire);
            }
            jedis.sadd(totalKey,value);
        } catch (Exception e){
            LOG.error("sadd the key error,db {} the dayKey is {} the value is {}",dbIndex, dayKey ,value, e);
        }finally {
            if (jedis != null) {
                jedis.close();
            }
        }
    }

    /**
     * 列表中删除与value相同的元素，返回删除的个数
     *
     * @param key
     * @param value
     * @return long
     * @throws
     * @author kk.xie
     * @date 2018/10/24 17:52
     */
    public long lrem(String key, String value){
        return lrem(key, value, null);
    }

    /**
     * 列表中删除与value相同的元素，返回删除的个数
     *
     * @param key
     * @param value
     * @param dbIndex
     * @return long
     * @throws
     * @author kk.xie
     * @date 2018/10/24 17:51
     */
    public long lrem(String key, String value, Integer dbIndex){
        Jedis jedis = null;
        try {
            jedis = jedisPool.getResource();
            if (dbIndex != null) {
                jedis.select(dbIndex);
            }
            return jedis.lrem(key, -1, value);
        } catch (Exception e){
            if(LOG.isErrorEnabled()){
                LOG.error("lrem the key error,db {} the key is {} the value is {}",dbIndex, key ,value, e);
            }
        }finally {
            if (jedis != null) {
                jedis.close();
            }
        }
        return 0L;
    }

    /**
     * 推送list
     *
     * @param key
     * @param value
     * @return void
     * @throws
     * @author kk.xie
     * @date 2018/10/24 16:31
     */
    public void rpush(String key, String value){
        rpush(key, value, null);
    }

    /**
     * 推送list
     *
     * @param key
     * @param value
     * @param dbIndex
     * @return void
     * @throws
     * @author kk.xie
     * @date 2018/10/24 16:31
     */
    public void rpush(String key, String value, Integer dbIndex){
        Jedis jedis = null;
        try {
            jedis = jedisPool.getResource();
            if (dbIndex != null) {
                jedis.select(dbIndex);
            }
            jedis.rpush(key, value);
        } catch (Exception e){
            if(LOG.isErrorEnabled()){
                LOG.error("rpush the key error,db {} the key is {} the value is {}",dbIndex, key ,value, e);
            }
        }finally {
            if (jedis != null) {
                jedis.close();
            }
        }
    }

    /**
     * 放到队列首端
     *
     * @param key
     * @param value
     * @return void
     * @throws
     * @author kk.xie
     * @date 2018/10/25 17:56
     */
    public void lpush(String key, String value){
        lpush(key, value, null);
    }

    /**
     * 放到队列首端
     *
     * @param key
     * @param value
     * @param dbIndex
     * @return void
     * @throws
     * @author kk.xie
     * @date 2018/10/25 17:56
     */
    public void lpush(String key, String value, Integer dbIndex){
        Jedis jedis = null;
        try {
            jedis = jedisPool.getResource();
            if (dbIndex != null) {
                jedis.select(dbIndex);
            }
            jedis.lpush(key, value);
        } catch (Exception e){
            if(LOG.isErrorEnabled()){
                LOG.error("lpush the key error,db {} the key is {} the value is {}",dbIndex, key ,value, e);
            }
        }finally {
            if (jedis != null) {
                jedis.close();
            }
        }
    }

    /**
     * 从左到右取 类似栈
     *
     * @param key
     * @return java.lang.String
     * @throws
     * @author kk.xie
     * @date 2018/10/24 16:33
     */
    public String lpop(String key){
        return lpop(key, null);
    }
    /**
     * 从左到右取 类似栈
     *
     * @param key
     * @param dbIndex
     * @return java.lang.String
     * @throws
     * @author kk.xie
     * @date 2018/10/24 16:33
     */
    public String lpop(String key, Integer dbIndex){
        Jedis jedis = null;
        try {
            jedis = jedisPool.getResource();
            if (dbIndex != null) {
                jedis.select(dbIndex);
            }
            return jedis.lpop(key);
        } catch (Exception e){
            if(LOG.isErrorEnabled()){
                LOG.error("lpop the key error,db {} the key is {} the value is {}",dbIndex, key , e);
            }
        }finally {
            if (jedis != null) {
                jedis.close();
            }
        }
        return null;
    }

    /**
     * list 长度
     *
     * @param key
     * @return long
     * @throws
     * @author kk.xie
     * @date 2018/10/25 21:49
     */
    public long llen(String key){
        return llen(key, null);
    }

    /**
     * list 长度
     *
     * @param key
     * @param dbIndex
     * @return long
     * @throws
     * @author kk.xie
     * @date 2018/10/25 21:48
     */
    public long llen(String key, Integer dbIndex){
        Jedis jedis = null;
        try {
            jedis = jedisPool.getResource();
            if (dbIndex != null) {
                jedis.select(dbIndex);
            }
            return jedis.llen(key);
        } catch (Exception e){
            if(LOG.isErrorEnabled()){
                LOG.error("llen the key error,db {} the key is {} the value is {}",dbIndex, key , e);
            }
        }finally {
            if (jedis != null) {
                jedis.close();
            }
        }
        return 0;
    }

    /**
     * 取值 key-set
     *
     * @param key
     * @return java.lang.String
     * @throws
     * @author kk.xie
     * @date 2018/10/19 17:52
     */
    public String spop(String key){
        return spop(key, null);
    }

    /**
     * 取值 key-set
     *
     * @param dbIndex 数据库
     * @param key
     * @return java.lang.String
     * @throws
     * @author kk.xie
     * @date 2018/10/19 17:52
     */
    public String spop(String key, Integer dbIndex){
        Jedis jedis = null;
        try {
            jedis = jedisPool.getResource();
            if (dbIndex != null) {
                jedis.select(dbIndex);
            }
            return jedis.spop(key);
        } catch (Exception e){
            if(LOG.isErrorEnabled()){
                LOG.error("spop the key error,db {} the key is {} ",dbIndex, key, e);
            }
        }finally {
            if (jedis != null) {
                jedis.close();
            }
        }
        return null;
    }

    /**
     * 删除key
     *
     * @param key
     * @return void
     * @throws
     * @author kk.xie
     * @date 2018/10/22 14:55
     */
    public void del(String key){
        del(key, null);
    }

    /**
     * 删除key
     *
     * @param key
     * @param dbIndex
     * @return void
     * @throws
     * @author kk.xie
     * @date 2018/10/22 14:55
     */
    public void del(String key, Integer dbIndex){
        Jedis jedis = null;
        try {
            jedis = jedisPool.getResource();
            if (dbIndex != null) {
                jedis.select(dbIndex);
            }
            jedis.del(key);
        } catch (Exception e){
            if(LOG.isErrorEnabled()){
                LOG.error("del the key error,db {} the key is {} ",dbIndex, key, e);
            }
        } finally {
            if (jedis != null) {
                jedis.close();
            }
        }
    }

    /**
     * 返回set集合的数量
     * @date: 2019/1/10
     * @param key
     * @return java.lang.Long
     * @throws 
     */
    public Long scard(String key){
        return scard(key,null);
    }
    /**
     * 返回set集合的数量
     * @date: 2019/1/10
     * @param key
     * @param dbIndex
     * @return java.lang.Long
     * @throws 
     */
    public Long scard(String key,Integer dbIndex){
        Jedis jedis = null;
        try {
            jedis = jedisPool.getResource();
            if (dbIndex != null) {
                jedis.select(dbIndex);
            }
            return jedis.scard(key);
        } catch (Exception e){
            LOG.error("scard the key error,db {} the key is {} ",dbIndex, key, e);
        } finally {
            if (jedis != null) {
                jedis.close();
            }
        }
        return 0L;
    }
    /**
     * 返回set集合中的所有成员
     * @date: 2019/1/10
     * @param key
     * @return java.util.Set<java.lang.String>
     * @throws 
     */
    public Set<String> smembers(String key){
        return smembers(key,null);
    }
    /**
     * 返回set集合中的所有成员
     * @date: 2019/1/10
     * @param key
     * @param dbIndex
     * @return java.util.Set<java.lang.String>
     * @throws 
     */
    public Set<String> smembers(String key,Integer dbIndex){
        Jedis jedis = null;
        try {
            jedis = jedisPool.getResource();
            if (dbIndex != null) {
                jedis.select(dbIndex);
            }
            return jedis.smembers(key);
        } catch (Exception e){
            LOG.error("scard the key error,db {} the key is {} ",dbIndex, key, e);
        } finally {
            if (jedis != null) {
                jedis.close();
            }
        }
        return null;
    }
    /**
     * 判断元素在set中是否存在
     *
     * @param key redis-key
     * @param member 成员变量
     * @return boolean
     * @throws
     * @author kk.xie
     * @date 2018/10/22 14:56
     */
    public boolean sismember(String key, String member){
        return sismember(key, member, null);
    }

    /**
     * 判断元素在set中是否存在
     *
     * @param key
     * @param member
     * @param dbIndex
     * @return boolean
     * @throws
     * @author kk.xie
     * @date 2018/10/22 14:56
     */
    public boolean sismember(String key, String member, Integer dbIndex){
        Jedis jedis = null;
        try {
            jedis = jedisPool.getResource();
            if (dbIndex != null) {
                jedis.select(dbIndex);
            }
            return jedis.sismember(key, member);
        } catch (Exception e){
            LOG.error("sismember the key error,db {} the key is {} ",dbIndex, key, e);
        } finally {
            if (jedis != null) {
                jedis.close();
            }
        }
        return false;
    }

    /**
     * 移除set中的元素
     * @date: 2018/12/17
     * @param key
     * @param values
     * @return java.lang.Long
     * @throws 
     */
    public Long srem(String key,String ... values){
        return srem(null,key,values);
    }
    /**
     * 移除set中的元素
     * @date: 2018/12/17
     * @param key
     * @param key
     * @param values
     * @return java.lang.Integer
     * @throws 
     */
    public Long srem(Integer dbIndex,String key,String ... values){
        if(StringUtils.isBlank(key)){
            return 0L;
        }

        Jedis jedis = null;
        try {
            jedis = jedisPool.getResource();
            if (dbIndex != null && dbIndex.intValue()>0) {
                jedis.select(dbIndex);
            }
            if(jedis.exists(key)){
                LOG.warn("the key:{} is not exist!",key);
                return 0L;
            }
            return jedis.srem(key,values);
        } catch (Exception e){
            LOG.error("srem the key error,db {} the key is {} ",dbIndex, key, e);
        } finally {
            if (jedis != null) {
                jedis.close();
            }
        }
        return 0L;
    }
    /**
     * 返回给定集合的并集。不存在的集合 key 被视为空集
     * @date: 2018/12/14
     * @param keys
     * @return java.util.Set<java.lang.String>
     * @throws 
     */
    public Set<String> sunion(String ... keys){
        return sunion(null,keys);
    }
    /**
     * 返回给定集合的并集。不存在的集合 key 被视为空集
     * @date: 2018/12/14
     * @param dbIndex
     * @param keys
     * @return java.util.Set<java.lang.String>
     * @throws 
     */
    public Set<String> sunion(Integer dbIndex, String ... keys){
        Jedis jedis = null;
        try {
            jedis = jedisPool.getResource();
            if (dbIndex != null) {
                jedis.select(dbIndex);
            }
            return jedis.sunion(keys);
        } catch (Exception e){
            LOG.error("sunion the key error,db {} the key is {} ",dbIndex, keys, e);
        } finally {
            if (jedis != null) {
                jedis.close();
            }
        }
        return new HashSet<String>();
    }

}
