package com.bxm.warcar.cache.impls.redis;

import com.bxm.warcar.cache.*;
import com.bxm.warcar.cache.serialization.JSONSerialization;
import com.google.common.collect.Maps;
import org.apache.commons.lang.StringUtils;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;

import java.util.List;
import java.util.Map;
import java.util.Set;

public class JedisFetcher implements Fetcher {

    public static final String HASH_ALL_FIELD = "____ALL____";

    private final JedisPool jedisPool;
    private final JSONSerialization serialization;
    /**
     * 是否启用ALL_FIELD实现
     */
    private boolean enableAllField = true;
    private String allFieldName = HASH_ALL_FIELD;

    public JedisFetcher(JedisPool jedisPool) {
        this.jedisPool = jedisPool;
        this.serialization = new JSONSerialization();
    }

    @Override
    public <T> T fetch(KeyGenerator keyGenerator, DataExtractor<T> dataExtractor, Class<T> cls) {
        return fetch(keyGenerator, dataExtractor, cls, 0);
    }

    @Override
    public <T> T fetch(KeyGenerator keyGenerator, DataExtractor<T> dataExtractor, Class<T> cls, int expireTimeInSecond) {
        if (null == keyGenerator) {
            throw new NullPointerException("getKeyGenerator");
        }

        String key = keyGenerator.generateKey();
        if (StringUtils.isBlank(key)) {
            throw new NullPointerException("key");
        }

        Jedis jedis = null;
        try {
            jedis = jedisPool.getResource();

            String value = jedis.get(key);

            if (null != value) {
                return serialization.deserialize(value, cls);
            }

            T fromStore = null;
            if (null != dataExtractor) {
                fromStore = dataExtractor.extract();
            }

            if (null != fromStore) {
                jedis.set(key, serialization.serialize(fromStore));
                if (expireTimeInSecond > 0)
                    jedis.expire(key, expireTimeInSecond);
            }

            return fromStore;
        }
        finally {
            if (null != jedis) {
                jedis.close();
            }
        }
    }

    @Override
    public <T> List<T> fetchList(KeyGenerator keyGenerator, DataExtractor<List<T>> dataExtractor, Class<T> cls) {
        return fetchList(keyGenerator, dataExtractor, cls, 0);
    }

    @Override
    public <T> List<T> fetchList(KeyGenerator keyGenerator, DataExtractor<List<T>> dataExtractor, Class<T> cls, int expireTimeInSecond) {
        if (null == keyGenerator) {
            throw new NullPointerException("getKeyGenerator");
        }

        String key = keyGenerator.generateKey();
        if (StringUtils.isBlank(key)) {
            throw new NullPointerException("key");
        }

        Jedis jedis = null;
        try {
            jedis = jedisPool.getResource();

            String value = jedis.get(key);

            if (null != value) {
                return serialization.deserializeList(value, cls);
            }

            List<T> fromStore = null;
            if (null != dataExtractor) {
                fromStore = dataExtractor.extract();
            }

            if (null != fromStore) {
                jedis.set(key, serialization.serialize(fromStore));
                if (expireTimeInSecond > 0)
                    jedis.expire(key, expireTimeInSecond);
            }

            return fromStore;
        }
        finally {
            if (null != jedis) {
                jedis.close();
            }
        }
    }

    @Override
    public <T> T hfetch(KeyGenerator keyGenerator, String field, DataExtractor<T> dataExtractor, Class<T> cls) {
        return this.hfetch(keyGenerator, field, dataExtractor, cls, 0);
    }

    @Override
    public <T> T hfetch(KeyGenerator keyGenerator, String field, DataExtractor<T> dataExtractor, Class<T> cls, int expireTimeInSecond) {
        if (null == keyGenerator) {
            throw new NullPointerException("getKeyGenerator");
        }

        String key = keyGenerator.generateKey();
        if (StringUtils.isBlank(key)) {
            throw new NullPointerException("key");
        }
        if (StringUtils.isBlank(field)) {
            throw new NullPointerException("field");
        }

        Jedis jedis = null;
        try {
            jedis = jedisPool.getResource();

            String value = jedis.hget(key, field);

            if (null != value) {
                return serialization.deserialize(value, cls);
            }

            T fromStore = null;
            if (null != dataExtractor) {
                fromStore = dataExtractor.extract();
            }

            if (null != fromStore) {
                jedis.hset(key, field, serialization.serialize(fromStore));
                if (expireTimeInSecond > 0)
                    jedis.expire(key, expireTimeInSecond);
            }

            return fromStore;
        }
        finally {
            if (null != jedis) {
                jedis.close();
            }
        }
    }

    @Override
    public <T> Map<String, T> hfetchall(KeyGenerator keyGenerator, DataExtractor<Map<String, T>> dataExtractor, Class<T> cls) {
        return hfetchall(keyGenerator, dataExtractor, cls, 0);
    }

    /**
     * Jedis 的hgetall实现，为了解决hgetall命令的性能问题。在该命令上加了一层缓存。
     * <pre>
     *     String allValue = jedis.hget(key, "ALL_FIELD_NAME");
     *     if (null == allValue) {
     *         jedis.hgetall(key);
     *     }
     * </pre>
     * 这个功能需要{@link JedisUpdater}配合使用，当更新这个hash中的任意一个field值时，需要刷新这个<code>ALL_FIELD_NAME</code>对应的值，以保证数据的一致性。
     * <b>更新频次高的谨慎使用。</b>
     *
     * @param keyGenerator
     * @param dataExtractor
     * @param cls
     * @param expireTimeInSecond
     * @param <T>
     * @return
     */
    @Override
    public <T> Map<String, T> hfetchall(KeyGenerator keyGenerator, DataExtractor<Map<String, T>> dataExtractor, Class<T> cls, int expireTimeInSecond) {
        if (null == keyGenerator) {
            throw new NullPointerException("getKeyGenerator");
        }

        String key = keyGenerator.generateKey();
        if (StringUtils.isBlank(key)) {
            throw new NullPointerException("key");
        }

        Jedis jedis = null;
        try {
            jedis = jedisPool.getResource();

            Map<String, T> rst = Maps.newHashMap();

            Map<String, String> value = null;

            if (enableAllField) {
                String allValFromCache = jedis.hget(key, allFieldName);
                if (StringUtils.isNotBlank(allValFromCache)) {
                    return serialization.deserializeMap(allValFromCache, cls);
                }
            }

            value = jedis.hgetAll(key);

            if (null != value) {
                Set<Map.Entry<String, String>> entries = value.entrySet();
                for (Map.Entry<String, String> entry : entries) {
                    String json = entry.getValue();
                    T deserialize = serialization.deserialize(json, cls);
                    rst.put(entry.getKey(), deserialize);
                }
                return rst;
            }

            Map<String, T> fromStore = null;
            if (null != dataExtractor) {
                fromStore = dataExtractor.extract();
            }

            if (null != fromStore) {
                Set<Map.Entry<String, T>> entries = fromStore.entrySet();
                for (Map.Entry<String, T> entry : entries) {
                    jedis.hset(key, entry.getKey(), serialization.serialize(entry.getValue()));
                }
                if (expireTimeInSecond > 0)
                    jedis.expire(key, expireTimeInSecond);
            }

            return fromStore;
        }
        finally {
            if (null != jedis) {
                jedis.close();
            }
        }
    }

    public boolean isEnableAllField() {
        return enableAllField;
    }

    public void setEnableAllField(boolean enableAllField) {
        this.enableAllField = enableAllField;
    }

    public String getAllFieldName() {
        return allFieldName;
    }

    public void setAllFieldName(String allFieldName) {
        this.allFieldName = allFieldName;
    }
}
