package com.bxm.warcar.cache.impls.redis;

import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.cache.Updater;
import com.bxm.warcar.cache.serialization.JSONSerialization;
import com.google.common.base.Preconditions;
import org.apache.commons.lang.StringUtils;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;

/**
 * @author allen
 * @date 2017-12-04
 */
public class JedisUpdater implements Updater {

    private final JedisPool jedisPool;
    private final JSONSerialization serialization;

    public JedisUpdater(JedisPool jedisPool) {
        this.jedisPool = jedisPool;
        this.serialization = new JSONSerialization();
    }

    @Override
    public void update(KeyGenerator keyGenerator, Object value, int expireTimeInSecond) {
        Preconditions.checkNotNull(keyGenerator);
        String key = keyGenerator.generateKey();
        if (StringUtils.isBlank(key)) {
            throw new NullPointerException("key");
        }
        if (null == value)
            throw new NullPointerException("value");

        Jedis jedis = null;
        try {
            jedis = jedisPool.getResource();

            jedis.set(key, this.serialization.serialize(value));
            if (expireTimeInSecond > 0)
                jedis.expire(key, expireTimeInSecond);
        }
        finally {
            if (null != jedis) {
                jedis.close();
            }
        }
    }

    @Override
    public void update(KeyGenerator keyGenerator, Object value) {
        this.update(keyGenerator, value, 0);
    }

    @Override
    public void remove(KeyGenerator keyGenerator) {
        Preconditions.checkNotNull(keyGenerator);
        String key = keyGenerator.generateKey();
        if (StringUtils.isBlank(key)) {
            throw new NullPointerException("key");
        }

        Jedis jedis = null;
        try {
            jedis = jedisPool.getResource();
            jedis.del(key);
        }
        finally {
            if (null != jedis) {
                jedis.close();
            }
        }
    }

    @Override
    public void hupdate(KeyGenerator keyGenerator, String field, Object value) {
        this.hupdate(keyGenerator, field, value, 0);
    }

    @Override
    public void hupdate(KeyGenerator keyGenerator, String field, Object value, int expireTimeInSecond) {
        Preconditions.checkNotNull(keyGenerator);
        String key = keyGenerator.generateKey();
        if (StringUtils.isBlank(key)) {
            throw new NullPointerException("key");
        }
        if (null == value)
            throw new NullPointerException("value");
        if (StringUtils.isBlank(field))
            throw new NullPointerException("field");

        Jedis jedis = null;
        try {
            jedis = jedisPool.getResource();

            jedis.hset(key, field, this.serialization.serialize(value));
            if (expireTimeInSecond > 0)
                jedis.expire(key, expireTimeInSecond);
        }
        finally {
            if (null != jedis) {
                jedis.close();
            }
        }
    }

    @Override
    public void hremove(KeyGenerator keyGenerator, String...field) {
        Preconditions.checkNotNull(keyGenerator);
        String key = keyGenerator.generateKey();
        if (StringUtils.isBlank(key)) {
            throw new NullPointerException("key");
        }

        Jedis jedis = null;
        try {
            jedis = jedisPool.getResource();

            jedis.hdel(key, field);
        }
        finally {
            if (null != jedis) {
                jedis.close();
            }
        }
    }
}
