package com.bxm.warcar.cache.impls.redis;

import com.bxm.warcar.cache.Counter;
import com.bxm.warcar.cache.DefaultValTrackable;
import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.utils.TypeHelper;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.math.NumberUtils;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;
import redis.clients.jedis.Transaction;

import java.util.List;

/**
 * @author allen
 * @date 2017-12-04
 */
public class JedisCounter implements Counter {

    private final JedisPool jedisPool;

    public JedisCounter(JedisPool jedisPool) {
        this.jedisPool = jedisPool;
    }

    @Override
    public Long incrementAndGet(KeyGenerator keyGenerator) {
        return incrementAndGet(keyGenerator, 0);
    }

    @Override
    public Long incrementAndGet(KeyGenerator keyGenerator, int expireTimeInSecond) {
        String key = getKey(keyGenerator);

        Jedis jedis = null;
        try {
            jedis = jedisPool.getResource();

            Long incr = jedis.incr(key);

            setExpire(jedis, key, expireTimeInSecond);

            return incr;
        }
        finally {
            if (null != jedis) {
                jedis.close();
            }
        }
    }

    @Override
    public Long incrementByAndGet(KeyGenerator keyGenerator, long inc) {
        return incrementByAndGet(keyGenerator, inc, 0);
    }

    @Override
    public Long incrementByAndGet(KeyGenerator keyGenerator, long inc, int expireTimeInSecond) {
        String key = getKey(keyGenerator);

        Jedis jedis = null;
        try {
            jedis = jedisPool.getResource();

            Long incr = jedis.incrBy(key, inc);

            setExpire(jedis, key, expireTimeInSecond);

            return incr;
        }
        finally {
            if (null != jedis) {
                jedis.close();
            }
        }
    }

    @Override
    public Long decrementAndGet(KeyGenerator keyGenerator) {
        return decrementAndGet(keyGenerator, 0);
    }

    @Override
    public Long decrementAndGet(KeyGenerator keyGenerator, int expireTimeInSecond) {
        String key = getKey(keyGenerator);

        Jedis jedis = null;
        try {
            jedis = jedisPool.getResource();

            Long decr = jedis.decr(key);

            setExpire(jedis, key, expireTimeInSecond);

            return decr;
        }
        finally {
            if (null != jedis) {
                jedis.close();
            }
        }
    }

    @Override
    public Long decrementByAndGet(KeyGenerator keyGenerator, long dec) {
        return decrementByAndGet(keyGenerator, dec, 0);
    }

    @Override
    public Long decrementByAndGet(KeyGenerator keyGenerator, long dec, int expireTimeInSecond) {
        String key = getKey(keyGenerator);

        Jedis jedis = null;
        try {
            jedis = jedisPool.getResource();

            Long decr = jedis.decrBy(key, dec);

            setExpire(jedis, key, expireTimeInSecond);

            return decr;
        }
        finally {
            if (null != jedis) {
                jedis.close();
            }
        }
    }

    @Override
    public void set(KeyGenerator keyGenerator, long value) {
        String key = getKey(keyGenerator);

        Jedis jedis = null;
        try {
            jedis = jedisPool.getResource();
            jedis.set(key, String.valueOf(value));
        }
        finally {
            if (null != jedis) {
                jedis.close();
            }
        }
    }

    @Override
    public Long get(KeyGenerator keyGenerator) {
        String key = getKey(keyGenerator);

        Jedis jedis = null;
        try {
            jedis = jedisPool.getResource();

            return TypeHelper.castToLong(jedis.get(key));
        }
        finally {
            if (null != jedis) {
                jedis.close();
            }
        }
    }

    @Override
    public Long hincrementAndGet(KeyGenerator keyGenerator, String field) {
        return hincrementAndGet(keyGenerator, field, 0);
    }

    @Override
    public Long hincrementAndGet(KeyGenerator keyGenerator, String field, int expireTimeInSecond) {
        return hincrementByAndGet(keyGenerator, field, 1, expireTimeInSecond);
    }

    @Override
    public Long hincrementByAndGet(KeyGenerator keyGenerator, String field, long inc) {
        return hincrementByAndGet(keyGenerator, field, inc, 0);
    }

    @Override
    public Long hincrementByAndGet(KeyGenerator keyGenerator, String field, long inc, int expireTimeInSecond) {
        String key = getKey(keyGenerator);

        Jedis jedis = null;
        try {
            jedis = jedisPool.getResource();

            Long decr = jedis.hincrBy(key, field, inc);

            setExpire(jedis, key, expireTimeInSecond);

            return decr;
        }
        finally {
            if (null != jedis) {
                jedis.close();
            }
        }
    }

    @Override
    public Long decrementAndGet(KeyGenerator keyGenerator, DefaultValTrackable defaultValTracker, long min, int expireTimeInSecond) {
        String key = getKey(keyGenerator);

        Jedis jedis = null;
        try {
            jedis = jedisPool.getResource();
            jedis.watch(key);

            String oldValue = jedis.get(key);
            if (null == oldValue) {
                // 初始化
                long start = defaultValTracker.getDefaultVal();

                Transaction t1 = jedis.multi();
                String startValue = String.valueOf(start);
                jedis.set(key, startValue);
                t1.set(key, startValue);
                List<Object> r1 = t1.exec();
                if (! isFinished(r1)) {
                    throw new IllegalStateException("Cannot initialized key-value for :" + key);
                }

                oldValue = startValue;

                // 事务执行完后已经unwatch，需要重新watch
                jedis.watch(key);
            }

            long old = NumberUtils.toLong(oldValue, min);
            if (old-- < min) {
                // 数值 -1 之后小于最小值
                throw new IndexOutOfBoundsException();
            }

            Transaction transaction = jedis.multi();
            transaction.set(key, String.valueOf(old));

            List<Object> result = transaction.exec(); // exec = unwatch
            boolean failed = isFinished(result);
            if (failed) {
                throw new IllegalStateException();
            }

            setExpire(jedis, key, expireTimeInSecond);

            return old;
        }
        finally {
            if (null != jedis) {
                jedis.unwatch();
                jedis.close();
            }
        }
    }

    private void setExpire(Jedis jedis, String key, int expireTimeInSecond) {
        if (expireTimeInSecond > 0)
            jedis.expire(key, expireTimeInSecond);
    }

    private boolean isFinished(List<Object> result) {
        return null == result || null == result.get(0);
    }

    private String getKey(KeyGenerator keyGenerator) {
        if (null == keyGenerator) {
            throw new NullPointerException("getKeyGenerator");
        }

        String key = keyGenerator.generateKey();
        if (StringUtils.isBlank(key)) {
            throw new NullPointerException("key");
        }
        return key;
    }
}
