/*
 * Copyright 2016 bianxianmao.com All right reserved. This software is the confidential and proprietary information of
 * textile.com ("Confidential Information"). You shall not disclose such Confidential Information and shall use it only
 * in accordance with the terms of the license agreement you entered into with bianxianmao.com.
 */

package com.bxm.warcar.configure.spring;

import com.bxm.warcar.configure.EnvironmentLoader;
import com.bxm.warcar.configure.loader.EnvironmentLoaderFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.boot.context.event.ApplicationEnvironmentPreparedEvent;
import org.springframework.context.ApplicationListener;
import org.springframework.core.env.ConfigurableEnvironment;
import org.springframework.core.env.MutablePropertySources;
import org.springframework.core.env.PropertiesPropertySource;
import org.springframework.core.env.PropertySource;

import java.util.Iterator;
import java.util.List;
import java.util.Properties;

/**
 * <h3>SpringBoot environment prepared listener.</h3>
 * <p>
 *     First step: you need set properties about configure to application.properties
 * </p>
 * <pre>
 *     application.properties
 *
 *     configure.address=zookeeper://127.0.0.1
 *     configure.key=/config/application.yml,/config/application.properties
 * </pre>
 * <p>
 *     Second step: Add this listener to SpringApplication
 * </p>
 * <pre>
 * <code>@</code>SpringBootConfiguration
 * public class SpringBootEnvironmentLoaderTest {
 *
 *     public static void main(String[] args) {
 *          SpringApplication application = new SpringApplication(SpringBootEnvironmentLoaderTest.class);
 *          application.addListeners(new SpringBootEnvironmentListener());
 *          application.run(args);
 *     }
 * }
 * </pre>
 *
 * @author allen
 * @since V1.0.0 2017/12/21
 */
public final class SpringBootEnvironmentListener implements ApplicationListener<ApplicationEnvironmentPreparedEvent> {

    private static final Logger LOGGER = LoggerFactory.getLogger(SpringBootEnvironmentListener.class);

    private static final String ADDRESS = "configure.address";
    private static final String KEY = "configure.key";

    @Override
    public void onApplicationEvent(ApplicationEnvironmentPreparedEvent event) {
        ConfigurableEnvironment environment = event.getEnvironment();
        MutablePropertySources propertySources = environment.getPropertySources();
        if (propertySources != null) {
            Iterator<PropertySource<?>> iterator = propertySources.iterator();
            while (iterator.hasNext()) {
                PropertySource<?> ps = iterator.next();

                boolean contains = ps.containsProperty(ADDRESS);
                if (! contains) {
                    continue;
                }
                Object address = ps.getProperty(ADDRESS);
                Object key = ps.getProperty(KEY);
                if (null == address || null == key) {
                    if (LOGGER.isWarnEnabled()) {
                        LOGGER.warn("Environment property '{}','{}' must not be null.", ADDRESS, KEY);
                    }
                    continue;
                }

                if (LOGGER.isInfoEnabled()) {
                    LOGGER.info("Loading properties '{}' from {}", key, address);
                }

                EnvironmentLoader environmentLoader = EnvironmentLoaderFactory.create(address.toString(), key.toString());
                List<Properties> properties = environmentLoader.loadProperties();

                closeLoader(environmentLoader);

                if (null != properties) {
                    for (int i = 0; i < properties.size(); i++) {
                        propertySources.addLast(new PropertiesPropertySource("userRemoteDefinedProperties-" + i, properties.get(i)));
                    }
                }
            }
        }
    }

    private void closeLoader(EnvironmentLoader environmentLoader) {
        try {
            environmentLoader.destroy();
        } catch (Exception e) {
            ;
        }
    }
}
